////////////////////////////////////////////////////////////////////////////////
/// @file udp.c
/// Ethernet, IP, ARP and ICMP packet handler functions.
///
/// @author  Tamas Raikovich
/// @version 1.0
/// @date    2012.01.21.
////////////////////////////////////////////////////////////////////////////////
#include <string.h>
#include <stdio.h>
#include "netif.h"
#include "dhcp_client.h"
#include "udp.h"


////////////////////////////////////////////////////////////////////////////////
///@name Function prototypes.
///@{
////////////////////////////////////////////////////////////////////////////////
unsigned long process_arp(pnetif_data netif, void *packet, unsigned long packetSize);
unsigned long process_ip(pnetif_data netif, void *packet, unsigned long packetSize);
unsigned long process_icmp(void *packet, unsigned long packetLength);
unsigned long process_udp(pnetif_data netif, pip_header ipHeader, void *packet, unsigned long packetLength);

void ProcessUDPData(pudp_conn conn, void *data, unsigned long *dataLength);
///@}


////////////////////////////////////////////////////////////////////////////////
/// Calculates the Internet checksum according to the RFC 1071.
///
/// @param sum    Initial value of the checksum.
/// @param data   Pointer to the data.
/// @param length Size of the data in bytes.
///
/// @return Calculated checksum.
////////////////////////////////////////////////////////////////////////////////
unsigned short checksum(unsigned long sum, void *data, unsigned long length)
{
    unsigned long add = 0;
    unsigned char *chkdata = (unsigned char *)data;

    for(; length > 1; length -= 2)
    {
        add = (unsigned long)((((unsigned short) *chkdata) << 8) + chkdata[1]);
        sum = sum + add;
        chkdata += 2;
    }

    if (length == 1)
        sum += (unsigned long)(((unsigned short) *chkdata) << 8);

    while (sum >> 16)
        sum = (sum & 0xffff) + (sum >> 16);

    return (unsigned short)sum;
}


////////////////////////////////////////////////////////////////////////////////
/// Calculates the IP checksum.
///
/// @param hdr Pointer to the IP header.
///
/// @return Calculated IP checksum.
////////////////////////////////////////////////////////////////////////////////
unsigned short ip_checksum(pip_header hdr)
{
    unsigned short sum;

    sum = ~checksum(0, hdr, sizeof(ip_header));
    return (sum == 0) ? 0xffff : HTONS(sum);
}


////////////////////////////////////////////////////////////////////////////////
/// Calculates the ICMP checksum.
///
/// @param hdr Pointer to the ICMP header.
/// @param len Size of the ICMP data in bytes.
///
/// @return Calculated ICMP checksum.
////////////////////////////////////////////////////////////////////////////////
unsigned short icmp_checksum(picmp_header hdr, unsigned long len)
{
    unsigned short sum;

    sum = ~checksum(0, hdr, len);
    return (sum == 0) ? 0xffff : HTONS(sum);
}


////////////////////////////////////////////////////////////////////////////////
/// Calculates the UDP checksum.
///
/// @param hdr Pointer to the IP header.
///
/// @return Calculated UDP checksum.
////////////////////////////////////////////////////////////////////////////////
unsigned short udp_checksum(pip_header hdr)
{
	unsigned short sum;
	unsigned short len   = NTOHS(hdr->packetlength) - sizeof(ip_header);
	unsigned char  *data = (unsigned char *)hdr + sizeof(ip_header);

	//IP protocol and length.
	sum = hdr->protocol + len;
	//Source and destination addresses.
	sum = checksum(sum, hdr->sourceip, 8);
	//UDP header and data.
	sum = ~checksum(sum, data, len);

	return (sum == 0) ? 0xffff : HTONS(sum);
}


////////////////////////////////////////////////////////////////////////////////
/// Processes the received Ethernet packet.
///
/// @param netif Pointer to the network interface object.
////////////////////////////////////////////////////////////////////////////////
void process_ethernet(pnetif_data netif)
{
    unsigned long length = netif->rxbytes;
    peth_frame eth = (peth_frame)(netif->packet);

    //Set the packet length to zero.
    netif->txbytes = 0;

    //Check the packet size.
    if (length < sizeof(eth_header))
        return;

    length -= sizeof(eth_header);

    //Check the frame type.
    switch (eth->header.length_type)
    {
        case HTONS(ETH_TYPE_ARP):
            length = process_arp(netif, eth->data, length);
            break;
        case HTONS(ETH_TYPE_IP):
            length = process_ip(netif, eth->data, length);
            break;
        default:
            return;
    }

    if (length > 0)
    {
        //Set the MAC addresses.
        memcpy(eth->header.destMAC, eth->header.sourceMAC, 6);
        memcpy(eth->header.sourceMAC, netif->macaddr, 6);

        //Set the packet length.
        netif->txbytes = length + sizeof(eth_header);
    }
}


////////////////////////////////////////////////////////////////////////////////
/// Processes the received ARP frame.
///
/// @param netif      Pointer to the network interface object.
/// @param packet     Pointer to the received data.
/// @param packetSize Size of the received data in bytes.
///
/// @return Size of the data to send in bytes.
////////////////////////////////////////////////////////////////////////////////
unsigned long process_arp(pnetif_data netif, void *packet, unsigned long packetSize)
{
    parp_header arp = (parp_header)packet;

    //Check the packet size.
    if (packetSize < sizeof(arp_header))
        return 0;

    //Check the request type.
    if (arp->operation != HTONS(ARP_REQUEST))
        return 0;

    //The target IP address cannot be 0.0.0.0
    if ((arp->targetprotocol[0] == 0) &&
        (arp->targetprotocol[1] == 0) &&
        (arp->targetprotocol[2] == 0) &&
        (arp->targetprotocol[3] == 0))
    {
        return 0;
    }

    //Compare the received IP address with our own IP address.
    if ((arp->targetprotocol[0] != netif->ipaddr[0]) ||
        (arp->targetprotocol[1] != netif->ipaddr[1]) ||
        (arp->targetprotocol[2] != netif->ipaddr[2]) ||
        (arp->targetprotocol[3] != netif->ipaddr[3]))
    {
        return 0;
    }

    //Check the remaining fields in the ARP header.
    if ((arp->hardware != HTONS(ARP_HARDWARE)) || (arp->protocol != HTONS(ETH_TYPE_IP)) ||
        (arp->hardwarelength != 0x06)   || (arp->protocollength != 0x04))
    {
        return 0;
    }

    //Send an ARP reply.
    arp->operation = HTONS(ARP_REPLY);

    //The target is the original sender.
    memcpy(arp->targethardware, arp->senderhardware, 6);
    memcpy(arp->targetprotocol, arp->senderprotocol, 4);

    //Copy our own MAC and IP addresses.
    memcpy(arp->senderhardware, netif->macaddr, 6);
    memcpy(arp->senderprotocol, netif->ipaddr, 4);

    return sizeof(arp_header);
}


////////////////////////////////////////////////////////////////////////////////
/// Processes the received IP packet.
///
/// @param netif      Pointer to the network interface object.
/// @param packet     Pointer to the received data.
/// @param packetSize Size of the received data in bytes.
///
/// @return Size of the data to send in bytes.
////////////////////////////////////////////////////////////////////////////////
unsigned long process_ip(pnetif_data netif, void *packet, unsigned long packetSize)
{
    pip_header ip = (pip_header)packet;
    void *ipdata;
    unsigned long ipdatalen = 0;
    unsigned long length = 0;

    //Check the frame size.
    if (packetSize < sizeof(ip_header))
        return 0;

    //Check the IP address.
    if ((netif->use_dhcp == 0) || (netif->dhcp_state.state == DHCP_STATE_CONFIG_RECEIVED))
    {
        if ((ip->destip[0] != netif->ipaddr[0]) ||
            (ip->destip[1] != netif->ipaddr[1]) ||
            (ip->destip[2] != netif->ipaddr[2]) ||
            (ip->destip[3] != netif->ipaddr[3]))
            return 0;
    }

    //Check the version.
    if ((ip->ver_hlen >> 4) != IP_VERSION_IP4)
        return 0;

    //Verify the IP checksum.
    if (ip_checksum(ip) != 0xffff)
        return 0;

    //Check the data size.
    ipdatalen = NTOHS(ip->packetlength) - sizeof(ip_header);
    if (ipdatalen > (packetSize - sizeof(ip_header)))
        return 0;

    ipdata = (unsigned char *)packet + sizeof(ip_header);

    //Check the protocol type.
    switch (ip->protocol)
    {
        case IP_PROTOCOL_ICMP:
            length = process_icmp(ipdata, ipdatalen);
            break;
        case IP_PROTOCOL_UDP:
        	length = process_udp(netif, ip, ipdata, ipdatalen);
        	break;
        default:
            return 0;
    }

    //Return to the caller if there is no data to send.
    if (length == 0)
        return 0;

    //Set the TTL value.
    ip->timetolive = 255;

    //The length of the IP packet.
    ip->packetlength = HTONS((unsigned short)(length + sizeof(ip_header)));

    //Change the IP address.
    memcpy(ip->destip, ip->sourceip, 4);
    memcpy(ip->sourceip, netif->ipaddr, 4);

    //Recalculate the UDP checksum if necessary.
    if (ip->protocol == IP_PROTOCOL_UDP)
    {
    	((pudp_header)ipdata)->checksum = 0;
    	((pudp_header)ipdata)->checksum = udp_checksum(ip);
    }

    //Recalculate the IP checksum.
    ip->checksum = 0;
    ip->checksum = ip_checksum(ip);

    return (length + sizeof(ip_header));
}


////////////////////////////////////////////////////////////////////////////////
/// Processes the received ICMP packet.
///
/// @param packet     Pointer to the received data.
/// @param packetSize Size of the received data in bytes.
///
/// @return Size of the data to send in bytes.
////////////////////////////////////////////////////////////////////////////////
unsigned long process_icmp(void *packet, unsigned long packetSize)
{
    picmp_header icmp = (picmp_header)packet;

    //Verify the checksum.
    if (icmp_checksum(icmp, packetSize) != 0xffff)
        return 0;

    //Examine the type of the ICMP packet.
    switch (icmp->type)
    {
        case ICMP_TYPE_ECHO:
            switch (icmp->code)
            {
                case 0:
                    //Send a reply.
                    icmp->type = ICMP_TYPE_ECHOREPLY;
                    break;
                default:
                    return 0;
            }
            break;
        default:
            return 0;
    }

    //Recalculate the checksum.
    icmp->checksum = 0;
    icmp->checksum = icmp_checksum(icmp, packetSize);

    return packetSize;
}


////////////////////////////////////////////////////////////////////////////////
/// Processes the received UDP packet.
///
/// @param netif      Pointer to the network interface object.
/// @param ipHeader   Pointer to the IP header.
/// @param packet     Pointer to the received data.
/// @param packetSize Size of the received data in bytes.
///
/// @return Size of the data to send in bytes.
////////////////////////////////////////////////////////////////////////////////
unsigned long process_udp(pnetif_data netif, pip_header ipHeader, void *packet, unsigned long packetSize)
{
    pudp_header udp = (pudp_header)packet;
    unsigned long udpdatalen;
    udp_conn conn;
    unsigned short port;

    //Verify the UDP checksum.
    if ((udp->checksum != 0) && (udp_checksum(ipHeader) != 0xffff))
    {
        return 0;
    }

    //Check the data size.
    udpdatalen = NTOHS(udp->length) - sizeof(udp_header);
    if (udpdatalen > (packetSize - sizeof(udp_header)))
    {
        return 0;
    }

    //Fill in the UDP connection structure.
    conn.sourceip[0] = ipHeader->sourceip[0];
    conn.sourceip[1] = ipHeader->sourceip[1];
    conn.sourceip[2] = ipHeader->sourceip[2];
    conn.sourceip[3] = ipHeader->sourceip[3];
    conn.sourceport = NTOHS(udp->sourceport);
    conn.destport = NTOHS(udp->destport);

    if ((udp->destport == HTONS(DHCPC_CLIENT_PORT)) && (udp->sourceport == HTONS(DHCPC_SERVER_PORT)))
    {
        process_dhcp(netif, udpdatalen, DHCP_FLAG_NEW_DATA);
        udpdatalen = 0;
    }
    else
    {
        //Call the application.
        ProcessUDPData(&conn, (unsigned char *)packet + sizeof(udp_header), &udpdatalen);
    }

    //Return to the caller if there is no data to send.
    if (udpdatalen == 0)
    {
        return 0;
    }

    //Swap the UDP source and destination ports.
    port = udp->sourceport;
    udp->sourceport = udp->destport;
    udp->destport = port;

    //Set the data length.
    udp->length = HTONS(udpdatalen + sizeof(udp_header));

    return (udpdatalen + sizeof(udp_header));
}
