////////////////////////////////////////////////////////////////////////////////
/// @file udp.h
/// Header file for the Ethernet, IP, ARP and ICMP packet handler functions.
///
/// @author  Tamas Raikovich
/// @version 1.0
/// @date    2012.01.21.
////////////////////////////////////////////////////////////////////////////////
#ifndef UDP_H_
#define UDP_H_

#include "netif.h"


////////////////////////////////////////////////////////////////////////////////
///@name Constants.
///@{
////////////////////////////////////////////////////////////////////////////////
#define ETH_TYPE_ARP            0x0806	///< Ethernet ARP frame.
#define ETH_TYPE_IP             0x0800	///< Ethernet IP frame.

#define ARP_HARDWARE            0x0001	///< ARP hardware address is the MAC address.
#define ARP_REQUEST             0x01	///< ARP request.
#define ARP_REPLY               0x02	///< ARP reply.

#define IP_VERSION_IP4          4		///< IP version 4.
#define IP_PROTOCOL_ICMP        1		///< ICMP protocol.
#define IP_PROTOCOL_UDP         17		///< UDP protocol.

#define ICMP_TYPE_ECHOREPLY     0x00	///< ICMP echo reply.
#define ICMP_TYPE_ECHO          0x08	///< ICMP echo request.
///@}


////////////////////////////////////////////////////////////////////////////////
/// Ethernet header.
////////////////////////////////////////////////////////////////////////////////
typedef struct _eth_header
{
    unsigned char  destMAC[6];          ///< Destination MAC address.
    unsigned char  sourceMAC[6];        ///< Source MAC address.
    unsigned short length_type;         ///< Packet type/length.
} eth_header, *peth_header;


////////////////////////////////////////////////////////////////////////////////
/// Ethernet frame without the CRC field.
////////////////////////////////////////////////////////////////////////////////
typedef struct _eth_frame
{
    eth_header    header;               ///< Ethernet header.
    unsigned char data[];               ///< Data.
} eth_frame, *peth_frame;


////////////////////////////////////////////////////////////////////////////////
/// ARP header.
////////////////////////////////////////////////////////////////////////////////
typedef struct _arp_header
{
    unsigned short hardware;            ///< Hardware ID.
    unsigned short protocol;            ///< Protocol ID.
    unsigned char  hardwarelength;      ///< Length of the hardware address.
    unsigned char  protocollength;      ///< Length of the protocol address.
    unsigned short operation;           ///< ARP operation.
    unsigned char  senderhardware[6];   ///< Hardware address of the sender.
    unsigned char  senderprotocol[4];   ///< Protocol address of the sender.
    unsigned char  targethardware[6];   ///< Hardware address of the target.
    unsigned char  targetprotocol[4];   ///< Protocol address of the target.
} arp_header, *parp_header;


////////////////////////////////////////////////////////////////////////////////
/// IPv4 header.
////////////////////////////////////////////////////////////////////////////////
typedef struct
{
    unsigned char  ver_hlen;            ///< Protocol version and header length.
    unsigned char  service;             ///< Type of service.
    unsigned short packetlength;        ///< Total length (IP header + data)
    unsigned short ident;               ///< IP fragments identification.
    unsigned short flags_offset;        ///< Flags and IP fragment offset.
    unsigned char  timetolive;          ///< Time to live.
    unsigned char  protocol;            ///< Protocol used in the data.
    unsigned short checksum;            ///< IP checksum.
    unsigned char  sourceip[4];         ///< Source IP address.
    unsigned char  destip[4];           ///< Destination IP address.
} ip_header, *pip_header;


////////////////////////////////////////////////////////////////////////////////
/// ICMP header.
////////////////////////////////////////////////////////////////////////////////
typedef struct
{
    unsigned char  type;                ///< ICMP message type.
    unsigned char  code;                ///< ICMP message subtype.
    unsigned short checksum;            ///< Checksum.
    unsigned char  rest[4];             ///< Rest of the ICMP header.
} icmp_header, *picmp_header;


////////////////////////////////////////////////////////////////////////////////
/// UDP header.
////////////////////////////////////////////////////////////////////////////////
typedef struct
{
	unsigned short sourceport;         ///< UDP source port.
	unsigned short destport;           ///< UDP destination port.
	unsigned short length;             ///< Total length (UDP header + data).
	unsigned short checksum;           ///< UDP checksum.
} udp_header, *pudp_header;


////////////////////////////////////////////////////////////////////////////////
/// UDP connection data structure that is passed to the application.
////////////////////////////////////////////////////////////////////////////////
typedef struct
{
	unsigned char   sourceip[4];		///< IP address of the sender.
	unsigned short  sourceport;			///< Source UDP port.
	unsigned short  destport;			///< Destination UDP port.
} udp_conn, *pudp_conn;


////////////////////////////////////////////////////////////////////////////////
///@name Function prototypes.
///@{
////////////////////////////////////////////////////////////////////////////////
void process_ethernet(pnetif_data netif);
///@}


#endif /* UDP_H_ */

