//*****************************************************************************
//* CPLD interface peripheral (PLB version).                                  *
//* Written by   : Tamas Raikovich                                            *
//* Version      : 1.0                                                        *
//* Last modified: 2013.04.28.                                                *
//*****************************************************************************
module user_logic #(
   //Bus protocol parameters.
   parameter C_SPLB_CLK_FREQ_HZ = 100000000,
   parameter C_SLV_DWIDTH       = 32,
   parameter C_NUM_REG          = 3
) (
   //Bus protocol ports.
   input  wire                        Bus2IP_Clk,        //PLB clock signal.
   input  wire                        Bus2IP_Reset,      //PLB reset signal.
   input  wire [0 : C_SLV_DWIDTH-1]   Bus2IP_Data,       //PLB input data bus.
   input  wire [0 : C_SLV_DWIDTH/8-1] Bus2IP_BE,         //PLB byte enable signals.
   input  wire [0 : C_NUM_REG-1]      Bus2IP_RdCE,       //PLB register read enable signals.
   input  wire [0 : C_NUM_REG-1]      Bus2IP_WrCE,       //PLB register write enable signals.
   output reg  [0 : C_SLV_DWIDTH-1]   IP2Bus_Data,       //PLB output data bus.
   output wire                        IP2Bus_RdAck,      //PLB read acknowledge signal.
   output wire                        IP2Bus_WrAck,      //PLB write acknowledge signal.
   output wire                        IP2Bus_Error,      //PLB bus error signal.
   output reg                         irq,               //Interrupt request signal.
   
   //Button input.
   input  wire [2:0]                  btn_in,
   
   //CPLD interface ports.
   output wire                        cpld_jtagen,       //CPLD JTAG interface enable signal.
   output wire                        cpld_rstn,         //CPLD reset signal.
   output wire                        cpld_clk,          //CPLD clock signal.
   output wire                        cpld_load,         //CPLD load signal.
   output wire                        cpld_mosi,         //CPLD serial data output.
   input  wire                        cpld_miso          //CPLD serial data input.
);

//*****************************************************************************
//* Base 2 logarithm function.                                                *
//*****************************************************************************
function integer log2(input integer x);
   for (log2 = 0; x > 0; log2 = log2 + 1)
      x = x >> 1;
endfunction


//******************************************************************************
//* Clock and reset signals.                                                   *
//******************************************************************************
wire clk = Bus2IP_Clk;
wire rst = Bus2IP_Reset;

//CPLD reset signal.
(* iob = "force" *)
reg cpld_rstn_reg;

always @(posedge clk)
begin
   cpld_rstn_reg <= ~rst;
end

assign cpld_rstn   = cpld_rstn_reg;
assign cpld_jtagen = 1'b0;

//******************************************************************************
//* LED register (BASE+0x00, 8 bit, RW).                                       *
//******************************************************************************
reg  [7:0] led_reg;
wire       led_reg_wr = Bus2IP_WrCE[0] & (Bus2IP_BE == 4'b1000);

always @(posedge clk)
begin
   if (rst)
      led_reg <= 8'd0;
   else
      if (led_reg_wr)
         led_reg <= Bus2IP_Data[0:7];
end


//******************************************************************************
//* DISP1 register (BASE+0x01, 8 bit, RW).                                     *
//******************************************************************************
reg  [7:0] disp1_reg;
wire       disp1_reg_wr = Bus2IP_WrCE[0] & (Bus2IP_BE == 4'b0100);

always @(posedge clk)
begin
   if (rst)
      disp1_reg <= 8'd0;
   else
      if (disp1_reg_wr)
         disp1_reg <= Bus2IP_Data[8:15];
end


//******************************************************************************
//* DISP2 register (BASE+0x02, 8 bit, RW).                                     *
//******************************************************************************
reg  [7:0] disp2_reg;
wire       disp2_reg_wr = Bus2IP_WrCE[0] & (Bus2IP_BE == 4'b0010);

always @(posedge clk)
begin
   if (rst)
      disp2_reg <= 8'd0;
   else
      if (disp2_reg_wr)
         disp2_reg <= Bus2IP_Data[16:23];
end


//******************************************************************************
//* Clock divider for generating the 3200 Hz clock enable signal.              *
//******************************************************************************
localparam CLK_DIV_VALUE = C_SPLB_CLK_FREQ_HZ / 3200;
localparam CLK_DIV_WIDTH = log2(CLK_DIV_VALUE - 1);

reg  [CLK_DIV_WIDTH-1:0] clk_divider;
wire                     clk_divider_tc = (clk_divider == 0);

always @(posedge clk)
begin
   if (rst || clk_divider_tc)
      clk_divider <= CLK_DIV_VALUE - 1;
   else
      clk_divider <= clk_divider - 1;
end


//******************************************************************************
//* Generating the CPLD clock signal.                                          *
//******************************************************************************
reg cpld_clk_reg;
(* iob = "force" *)
reg cpld_clk_out_reg;

always @(posedge clk)
begin
   if (rst)
      cpld_clk_reg <= 1'b0;
   else
      if (clk_divider_tc)
         cpld_clk_reg <= ~cpld_clk_reg;
end

always @(posedge clk)
begin
   if (rst)
      cpld_clk_out_reg <= 1'b0;
   else
      cpld_clk_out_reg <= cpld_clk_reg;
end

assign cpld_clk = cpld_clk_out_reg;


//******************************************************************************
//* Generating the CPLD load signal.                                           *
//******************************************************************************
reg [3:0] cpld_clk_cnt;
reg       cpld_load_reg;
(* iob = "force" *)
reg       cpld_load_out_reg;

always @(posedge clk)
begin
   if (rst)
      cpld_clk_cnt <= 4'd0;
   else
      if (clk_divider_tc && cpld_clk_reg)
         cpld_clk_cnt <= cpld_clk_cnt + 4'd1;
end

always @(posedge clk)
begin
   if (rst)
      cpld_load_reg <= 1'b0;
   else
      cpld_load_reg <= (cpld_clk_cnt == 4'd15);
end

always @(posedge clk)
begin
   if (rst)
      cpld_load_out_reg <= 1'b0;
   else
      cpld_load_out_reg <= cpld_load_reg;
end

assign cpld_load = cpld_load_out_reg;


//******************************************************************************
//* Sampling the CPLD serial input data.                                       *
//******************************************************************************
(* iob = "force" *)
reg cpld_miso_in_reg;
reg cpld_miso_sample;

always @(posedge clk)
begin
   cpld_miso_in_reg <= cpld_miso;
end

always @(posedge clk)
begin
   if (clk_divider_tc && (cpld_clk_reg == 0))
      cpld_miso_sample <= cpld_miso_in_reg;
end


//******************************************************************************
//* Shift register.                                                            *
//******************************************************************************
reg  [15:0] shr;
wire [15:0] shr_din;
reg         disp_sel;
(* iob = "force" *)
reg         cpld_mosi_reg;

assign shr_din[7:0]  = led_reg;
assign shr_din[15:8] = (disp_sel) ? disp1_reg : disp2_reg;

always @(posedge clk)
begin
   if (rst)
      shr <= 16'd0;
   else
      if (clk_divider_tc && cpld_clk_reg)
         if (cpld_load_reg)
            shr <= shr_din;
         else
            shr <= {cpld_miso_sample, shr[15:1]};
end

always @(posedge clk)
begin
   if (rst)
      disp_sel <= 1'b0;
   else
      if (clk_divider_tc && cpld_clk_reg && cpld_load_reg)
         disp_sel <= shr[14];
end

always @(posedge clk)
begin
   cpld_mosi_reg <= shr[0];
end

assign cpld_mosi = cpld_mosi_reg;


//******************************************************************************
//* DIP switch register (BASE+0x04, 8 bit, R).                                 *
//******************************************************************************
reg [7:0] dipsw_reg;

always @(posedge clk)
begin
   if (rst)
      dipsw_reg <= 8'd0;
   else
      if (clk_divider_tc && cpld_clk_reg && cpld_load_reg)
         dipsw_reg <= shr[8:1];
end


//******************************************************************************
//* Navigation switch and buttons register (BASE+0x05, 8 bit, R).              *
//******************************************************************************
(* iob = "force" *)
reg [2:0] btn_in_reg;
reg [7:0] navsw_btn_reg;

always @(posedge clk)
begin
   btn_in_reg <= btn_in;
end

always @(posedge clk)
begin
   if (rst)
      navsw_btn_reg <= 8'd0;
   else
      if (clk_divider_tc && cpld_clk_reg && cpld_load_reg)
         navsw_btn_reg <= {btn_in_reg, shr[13:9]};
end


//******************************************************************************
//* Interrupt flag register (BASE+0x08, 16 bit, RW).                           *
//******************************************************************************
wire [15:0] input_data = {navsw_btn_reg, dipsw_reg};
reg  [15:0] input_samples;
wire [15:0] input_changed = input_data ^ input_samples;
reg  [15:0] if_reg;
wire        if_reg_wr = Bus2IP_WrCE[2] & (Bus2IP_BE == 4'b1100);

integer i;

always @(posedge clk)
begin
   if (rst)
      input_samples <= 16'd0;
   else
      input_samples <= input_data;
end

always @(posedge clk)
begin
   for (i = 0; i < 16; i = i + 1)
      if (rst)
         if_reg[i] <= 1'b0;
      else
         if (input_changed[i])
            if_reg[i] <= 1'b1;
         else
            if (if_reg_wr && Bus2IP_Data[15-i])
               if_reg[i] <= 1'b0;
end


//******************************************************************************
//* Interrupt enable register (BASE+0x0A, 16 bit, RW).                         *
//******************************************************************************
reg  [15:0] ie_reg;
wire        ie_reg_wr = Bus2IP_WrCE[2] & (Bus2IP_BE == 4'b0011);

always @(posedge clk)
begin
   if (rst)
      ie_reg <= 16'd0;
   else
      if (ie_reg_wr)
         ie_reg <= Bus2IP_Data[16:31];
end


//******************************************************************************
//* Interrupt request logic.                                                   *
//******************************************************************************
always @(posedge clk)
begin
   if (rst)
      irq <= 1'b0;
   else
      irq <= |(ie_reg & if_reg);
end 


//******************************************************************************
//* Driving the PLB output ports.                                              *
//******************************************************************************
assign IP2Bus_RdAck = |Bus2IP_RdCE;
assign IP2Bus_WrAck = |Bus2IP_WrCE;
assign IP2Bus_Error = 1'b0;

always @(*)
begin
   case (Bus2IP_RdCE)
      3'b100 : IP2Bus_Data <= {led_reg, disp1_reg, disp2_reg, 8'd0};
      3'b010 : IP2Bus_Data <= {dipsw_reg, navsw_btn_reg, 16'd0};
      3'b001 : IP2Bus_Data <= {if_reg, ie_reg};
      default: IP2Bus_Data <= 32'd0;
   endcase
end 
 

endmodule
