//******************************************************************************
//* XCL SDRAM controller for the LOGSYS Spartan-6 FPGA board.                  *
//*                                                                            *
//* Written by   : Tamas Raikovich                                             *
//* Version      : 1.0                                                         *
//* Last modified: 2012.10.21.                                                 *
//******************************************************************************
module logsys_xcl_sdram_ctrl #(
   //SDRAM timing parameters.
   parameter C_T_SDRAM_RP_NS       = 20,              //Wait time after PRECHARGE command in ns
   parameter C_T_SDRAM_RFC_NS      = 66,              //AUTO REFRESH command period in ns
   parameter C_T_SDRAM_RMD_CLK     = 2,               //Wait time after mode reg. write in clocks
   parameter C_T_SDRAM_RCD_NS      = 20,              //RAS to CAS delay in ns
   parameter C_T_SDRAM_RC_NS       = 66,              //RAS to RAS delay in ns
   parameter C_T_SDRAM_RAS_MIN_NS  = 42,              //Min. row active time in ns
   parameter C_T_SDRAM_RAS_MAX_NS  = 100000,          //Max. row active time in ns
   parameter C_T_SDRAM_REFRESH_MS  = 64,              //Refresh period in ms
   parameter C_SDRAM_REFRESH_BURST = 8,               //Number of issued AUTO REFRESH commands
   parameter C_SDRAM_CAS_LATENCY   = 2,               //CAS latency in clocks
   
   //XCL interface parameters.
   parameter C_MEM_BASEADDR        = 32'hffffffff,    //Base address
   parameter C_MEM_HIGHADDR        = 32'h00000000,    //High address
   parameter C_XCL0_LINESIZE       = 4,               //Cacheline size for XCL0
   parameter C_XCL0_IN_FIFO_DEPTH  = 16,              //Input fifo depth for XCL0
   parameter C_XCL0_OUT_FIFO_DEPTH = 16,              //Output fifo depth for XCL0
   parameter C_XCL1_LINESIZE       = 4,               //Cacheline size for XCL1
   parameter C_XCL1_IN_FIFO_DEPTH  = 16,              //Input fifo depth for XCL1
   parameter C_XCL1_OUT_FIFO_DEPTH = 16,              //Output fifo depth for XCL1
   
   //System clock period in ps.
   parameter C_SYSCLK_PERIOD_PS    = 10000
) ( 
   //Clock and reset.
   input  wire        clk,                            //1x system clock signal
   input  wire        clk2x,                          //2x system clock signal
   input  wire        rst,                            //Reset signal
   
   //XCL interface 0 signals.
   input  wire        xcl0_access_ctrl,               //XCL control input
   input  wire [0:31] xcl0_access_data,               //XCL input data
   input  wire        xcl0_access_wr,                 //XCL input FIFO write signal
   output wire        xcl0_access_full,               //XCL input FIFO is full
   output wire        xcl0_rddata_ctrl,               //XCL control output
   output wire [0:31] xcl0_rddata_data,               //XCL data output
   input  wire        xcl0_rddata_rd,                 //XCL output FIFO read signal
   output wire        xcl0_rddata_exists,             //XCL output FIFO is not empty
   
   //XCL interface 1 signals.
   input  wire        xcl1_access_ctrl,               //XCL control input
   input  wire [0:31] xcl1_access_data,               //XCL input data
   input  wire        xcl1_access_wr,                 //XCL input FIFO write signal
   output wire        xcl1_access_full,               //XCL input FIFO is full
   output wire        xcl1_rddata_ctrl,               //XCL control output
   output wire [0:31] xcl1_rddata_data,               //XCL data output
   input  wire        xcl1_rddata_rd,                 //XCL output FIFO read signal
   output wire        xcl1_rddata_exists,             //XCL output FIFO is not empty
   
   //Common memory interface signals.
   output wire [17:0] mem_addr,                       //Memory address bus
   output wire [15:0] mem_data_O,                     //Memory data bus
   input  wire [15:0] mem_data_I,
   output wire [15:0] mem_data_T,
   output wire        mem_wen,                        //Memory write enable signal (active low)
   output wire        mem_lbn,                        //Lower data byte enable (active low)
   output wire        mem_ubn,                        //Upper data byte enable (active low)
   
   //Signals related to the SRAM.
   output wire        sram_csn,                       //SRAM chip select signal (active low)
   output wire        sram_oen,                       //SRAM output enable signal (active low)
   
   //Signals related to the SDRAM.
   output wire        sdram_clk,                      //SDRAM clock signal
   output wire        sdram_cke,                      //SDRAM clock enable signal
   output wire        sdram_csn,                      //SDRAM chip select signal (active low)
   
   //SDRAM initialization done signal.
   output wire        init_done
);

`include "memctrl_defs.vh"
`include "functions.vh"

//******************************************************************************
//* XCL interface.                                                             *
//******************************************************************************
wire        mem_write_req;
wire        mem_read_req;
wire [24:1] mem_address;
wire [3:0]  mem_byte_en;
wire        mem_write_ack;
wire        mem_wr_valid;
wire [31:0] mem_wr_data;
wire        mem_read_ack;
wire        mem_rd_valid;
wire [31:0] mem_rd_data;

xcl_interface #(
   //XCL0 interface parameters.
   .XCL0_CACHELINE_SIZE(C_XCL0_LINESIZE),          //Cacheline size in 32-bit words
   .XCL0_IN_FIFO_DEPTH(C_XCL0_IN_FIFO_DEPTH),      //Depth of the XCL input FIFO
   .XCL0_OUT_FIFO_DEPTH(C_XCL0_OUT_FIFO_DEPTH),    //Depth of the XCL output FIFO
   
   //XCL0 interface parameters.
   .XCL1_CACHELINE_SIZE(C_XCL1_LINESIZE),          //Cacheline size in 32-bit words
   .XCL1_IN_FIFO_DEPTH(C_XCL1_IN_FIFO_DEPTH),      //Depth of the XCL input FIFO
   .XCL1_OUT_FIFO_DEPTH(C_XCL1_OUT_FIFO_DEPTH),    //Depth of the XCL output FIFO
   
   //Memory read pipeline delay in clocks.
   .MEM_RD_DELAY_CLK(C_SDRAM_CAS_LATENCY+1)
) xcl_interface (
   //Clock and reset.
   .clk(clk),                                      //Clock signal
   .rst(rst),                                      //Reset signal
   
   //CacheLink interface 0 signals (big-endian).
   .xcl0_access_ctrl(xcl0_access_ctrl),            //XCL control input
   .xcl0_access_data(xcl0_access_data),            //XCL input data
   .xcl0_access_wr(xcl0_access_wr),                //XCL input FIFO write signal
   .xcl0_access_full(xcl0_access_full),            //XCL input FIFO is full
   .xcl0_rddata_ctrl(xcl0_rddata_ctrl),            //XCL control output
   .xcl0_rddata_data(xcl0_rddata_data),            //XCL data output
   .xcl0_rddata_rd(xcl0_rddata_rd),                //XCL output FIFO read signal
   .xcl0_rddata_exists(xcl0_rddata_exists),        //XCL output FIFO is not empty
   
   //CacheLink interface 0 signals (big-endian).
   .xcl1_access_ctrl(xcl1_access_ctrl),            //XCL control input
   .xcl1_access_data(xcl1_access_data),            //XCL input data
   .xcl1_access_wr(xcl1_access_wr),                //XCL input FIFO write signal
   .xcl1_access_full(xcl1_access_full),            //XCL input FIFO is full
   .xcl1_rddata_ctrl(xcl1_rddata_ctrl),            //XCL control output
   .xcl1_rddata_data(xcl1_rddata_data),            //XCL data output
   .xcl1_rddata_rd(xcl1_rddata_rd),                //XCL output FIFO read signal
   .xcl1_rddata_exists(xcl1_rddata_exists),        //XCL output FIFO is not empty
   
   //Memory controller interface signals (little-endian).
   .mem_write_req(mem_write_req),                  //Write transfer request signal
   .mem_read_req(mem_read_req),                    //Read transfer request signal
   .mem_address(mem_address),                      //Address output
   .mem_byte_en(mem_byte_en),                      //Byte enable signals
   .mem_write_ack(mem_write_ack),                  //Write acknowledge signal
   .mem_wr_valid(mem_wr_valid),                    //Write data valid signal
   .mem_wr_data(mem_wr_data),                      //Write data output
   .mem_read_ack(mem_read_ack),                    //Read acknowledge signal
   .mem_rd_valid(mem_rd_valid),                    //Read data valid signal
   .mem_rd_data(mem_rd_data)                       //Read data input
);


//******************************************************************************
//* Memory controller state machine.                                           *
//******************************************************************************
wire        iodelay_cal;
wire        iodelay_busy;
wire [1:0]  sdram_bank_addr;
wire [12:0] sdram_row_addr;  
wire        sdram_cke_set;
wire [8:0]  sdram_cmd;
wire [1:0]  sdram_addr_sel;
wire        dout_buf_tri;
wire        be_valid;
wire        be_delay_sel;

sdram_ctrl #(   
   //SDRAM timing parameters.
   .SDRAM_T_RP_PS(C_T_SDRAM_RP_NS * 1000),            //Wait time after PRECHARGE command in ps
   .SDRAM_T_RFC_PS(C_T_SDRAM_RFC_NS * 1000),          //AUTO REFRESH command period in ps
   .SDRAM_T_RMD_CLK(C_T_SDRAM_RMD_CLK),               //Wait time after mode reg. write in clocks
   .SDRAM_T_RCD_PS(C_T_SDRAM_RCD_NS * 1000),          //RAS to CAS delay in ps
   .SDRAM_T_RC_PS(C_T_SDRAM_RC_NS * 1000),            //RAS to RAS delay in ps
   .SDRAM_T_RAS_MIN_PS(C_T_SDRAM_RAS_MIN_NS * 1000),  //Min. row active time in ps
   .SDRAM_T_RAS_MAX_PS(C_T_SDRAM_RAS_MAX_NS * 1000),  //Max. row active time in ps
   .SDRAM_T_REFRESH_MS(C_T_SDRAM_REFRESH_MS),         //Refresh period in ms
   .SDRAM_REFRESH_BURST(C_SDRAM_REFRESH_BURST),       //Number of issued AUTO REFRESH commands
   .SDRAM_CAS_LATENCY(C_SDRAM_CAS_LATENCY),           //CAS latency in clocks
   
   //System clock parameters.
   .SYSCLK_PERIOD_PS(C_SYSCLK_PERIOD_PS)              //System clock period in ps
) sdram_ctrl (
   //Clock and reset.
   .clk(clk),                                         //Clock signal
   .rst(rst),                                         //Reset signal (active-low)
   
   //Input control signals.
   .mem_write_req(mem_write_req),                     //Memory write request signal
   .mem_read_req(mem_read_req),                       //Memory read request signal
   .wr_data_valid(mem_wr_valid),                      //Write data valid signal
   .sdram_bank_addr(sdram_bank_addr),                 //SDRAM bank address
   .sdram_row_addr(sdram_row_addr),                   //SDRAM row address
   .iodelay_busy(iodelay_busy),                       //IODELAY calibration is in progress
   
   //Output control signals.
   .sdram_cke_set(sdram_cke_set),                     //SDRAM clock enable set signal
   .sdram_init_done(init_done),                       //SDRAM initialization done signal
   .iodelay_cal(iodelay_cal),                         //IODELAY calibration start signal
   .sdram_cmd(sdram_cmd),                             //SDRAM command
   .sdram_addr_sel(sdram_addr_sel),                   //SDRAM address select signal
   .mem_write_ack(mem_write_ack),                     //Write acknowledge signal
   .mem_read_ack(mem_read_ack),                       //Read acknowledge signal
   .dout_buf_tri(dout_buf_tri),                       //Output buffer tri-state signal
   .be_valid(be_valid),                               //Byte enable valid signal
   .be_delay_sel(be_delay_sel),                       //Byte enable delay select  
   .rd_data_valid(mem_rd_valid)                       //Read data valid signal
);


//******************************************************************************
//* Physical interface.                                                        *
//******************************************************************************
localparam SDRAM_MREG_VALUE = (C_SDRAM_CAS_LATENCY == 3) ? SDRAM_MREG_VALUE_CL3 : SDRAM_MREG_VALUE_CL2;

phy phy(
   //Clock and reset.
   .clk(clk),                                         //1x system clock signal
   .clk2x(clk2x),                                     //2x system clock signal
   .rst(rst),                                         //Asynchronous reset signal
   
   //Signals from the memory controller.
   .sdram_cke_set(sdram_cke_set),                     //SDRAM clock enable output set signal
   .sdram_mreg_val(SDRAM_MREG_VALUE),                 //Value of the SDRAM mode register
   .sdram_cmd(sdram_cmd),                             //SDRAM command input
   .address_in(mem_address),                          //Memory address input
   .sdram_addr_sel(sdram_addr_sel),                   //SDRAM address select signal
   .write_data(mem_wr_data),                          //Write data input (little-endian)
   .dout_buf_tri(dout_buf_tri),                       //Output buffer tri-state signal  
   .byte_enable(mem_byte_en),                         //Byte enable signals (little-endian)
   .be_valid(be_valid),                               //Byte enable valid signal
   .be_delay_sel(be_delay_sel),                       //Byte enable delay select
   .iodelay_cal(iodelay_cal),                         //IODELAY calibration start signal
   .init_done(init_done),                             //SDRAM initialization done signal
   
   //Signals to the memory controller.
   .read_data(mem_rd_data),                           //Read data output (little-endian)
   .sdram_bank_addr(sdram_bank_addr),                 //SDRAM bank address
   .sdram_row_addr(sdram_row_addr),                   //SDRAM row address
   .sdram_col_addr(),                                 //SDRAM column address
   .iodelay_busy(iodelay_busy),                       //IODELAY calibration is in progress
   
   //Common memory interface signals.
   .mem_addr(mem_addr),                               //Memory address bus
   .mem_data_O(mem_data_O),                           //Memory data bus
   .mem_data_I(mem_data_I),
   .mem_data_T(mem_data_T),
   .mem_wen(mem_wen),                                 //Memory write enable signal (active low)
   .mem_lbn(mem_lbn),                                 //Lower data byte enable (active low)
   .mem_ubn(mem_ubn),                                 //Upper data byte enable (active low)
   
   //Signals related to the SRAM.
   .sram_csn(sram_csn),                               //SRAM chip select signal (active low)
   .sram_oen(sram_oen),                               //SRAM output enable signal (active low)
   
   //Signals related to the SDRAM.
   .sdram_clk(sdram_clk),                             //SDRAM clock signal
   .sdram_cke(sdram_cke),                             //SDRAM clock enable signal
   .sdram_csn(sdram_csn)                              //SDRAM chip select signal (active low)
);


endmodule
