////////////////////////////////////////////////////////////////////////////////
/// @file dhcp_client.c
/// DHCP client functions.
///
/// @author  Tamas Raikovich
/// @version 1.0
/// @date    2012.01.21.
////////////////////////////////////////////////////////////////////////////////
#include <string.h>
#include <stdio.h>
#include "netif.h"
#include "udp.h"
#include "timer.h"
#include "dhcp_client.h"


////////////////////////////////////////////////////////////////////////////////
/// DHCP message structure.
////////////////////////////////////////////////////////////////////////////////
typedef struct _dhcp_msg
{
    unsigned char  op;              ///< Message opcode/type.
    unsigned char  htype;           ///< Hardware address type.
    unsigned char  hlen;            ///< Hardware address length.
    unsigned char  hops;
    unsigned char  xid[4];          ///< Transaction ID.
    unsigned short secs;            ///< Seconds elapsed since client began acquisition/renewal.
    unsigned short flags;           ///< Flags.
    unsigned char  ciaddr[4];       ///< Client IP address.
    unsigned char  yiaddr[4];       ///< 'Your' (client) IP address.
    unsigned char  siaddr[4];       ///< Server IP address.
    unsigned char  giaddr[4];       ///< Relay agent IP address.
    unsigned char  chaddr[16];      ///< Client hardware address.
    unsigned char  sname[64];
    unsigned char  file[128];
    unsigned char  options[312];    ///< Optional parameters.
} dhcp_msg, *pdhcp_msg;


////////////////////////////////////////////////////////////////////////////////
///@name DHCP constants.
///@{
////////////////////////////////////////////////////////////////////////////////
//BOOTP broadcast flag.
#define BOOTP_BROADCAST             0x8000

//DHCP message opcodes.
#define DHCP_REQUEST                1
#define DHCP_REPLY                  2

//DHCP hardware address constants (Ethernet MAC address)
#define DHCP_HTYPE_ETHERNET         1
#define DHCP_HLEN_ETHERNET          6

//DHCP message type identifiers.
#define DHCPDISCOVER                1
#define DHCPOFFER                   2
#define DHCPREQUEST                 3
#define DHCPDECLINE                 4
#define DHCPACK                     5
#define DHCPNAK                     6
#define DHCPRELEASE                 7

//DHCP option identifiers.
#define DHCP_OPTION_SUBNET_MASK     1
#define DHCP_OPTION_ROUTER          3
#define DHCP_OPTION_DNS_SERVER      6
#define DHCP_OPTION_REQ_IPADDR      50
#define DHCP_OPTION_LEASE_TIME      51
#define DHCP_OPTION_MSG_TYPE        53
#define DHCP_OPTION_SERVER_ID       54
#define DHCP_OPTION_REQ_LIST        55
#define DHCP_OPTION_END             255

static const unsigned char xid[4] = {0xad, 0xde, 0x12, 0x23};
static const unsigned char magic_cookie[4] = {99, 130, 83, 99};
static const unsigned char destip[4] = {192, 168, 1, 10};
///@}


////////////////////////////////////////////////////////////////////////////////
///@name Function prototypes.
///@{
////////////////////////////////////////////////////////////////////////////////
unsigned short ip_checksum(pip_header hdr);
unsigned short udp_checksum(pip_header hdr);
///@}


////////////////////////////////////////////////////////////////////////////////
/// Adds the message type option to the DHCP message.
///
/// @param optptr Pointer to the DHCP message data.
/// @param type   DHCP message type.
////////////////////////////////////////////////////////////////////////////////
unsigned char *add_msg_type(unsigned char *optptr, unsigned char type)
{
    *optptr++ = DHCP_OPTION_MSG_TYPE;
    *optptr++ = 1;
    *optptr++ = type;
    return optptr;
}


////////////////////////////////////////////////////////////////////////////////
/// Adds the DHCP server IP address to the DHCP message.
///
/// @param optptr     Pointer to the DHCP message data.
/// @param dhcp_state Pointer to the DHCP data object.
////////////////////////////////////////////////////////////////////////////////
unsigned char *add_server_id(unsigned char *optptr, pdhcp_state dhcp_state)
{
    *optptr++ = DHCP_OPTION_SERVER_ID;
    *optptr++ = 4;
    memcpy(optptr, dhcp_state->serverid, 4);
    return optptr + 4;
}


////////////////////////////////////////////////////////////////////////////////
/// Adds the client's IP address to the DHCP message.
///
/// @param optptr     Pointer to the DHCP message data.
/// @param dhcp_state Pointer to the DHCP data object.
////////////////////////////////////////////////////////////////////////////////
unsigned char *add_req_ipaddr(unsigned char *optptr, pdhcp_state dhcp_state)
{
    *optptr++ = DHCP_OPTION_REQ_IPADDR;
    *optptr++ = 4;
    memcpy(optptr, dhcp_state->ipaddr, 4);
    return optptr + 4;
}


////////////////////////////////////////////////////////////////////////////////
/// Adds the requested informations to the DHCP message.
///
/// @param optptr Pointer to the DHCP message data.
////////////////////////////////////////////////////////////////////////////////
unsigned char *add_req_options(unsigned char *optptr)
{
    *optptr++ = DHCP_OPTION_REQ_LIST;
    *optptr++ = 3;
    *optptr++ = DHCP_OPTION_SUBNET_MASK;
    *optptr++ = DHCP_OPTION_ROUTER;
    *optptr++ = DHCP_OPTION_DNS_SERVER;
    return optptr;
}


////////////////////////////////////////////////////////////////////////////////
/// Adds the END marker to the DHCP message.
///
/// @param optptr Pointer to the DHCP message data.
////////////////////////////////////////////////////////////////////////////////
unsigned char *add_end(unsigned char *optptr)
{
    *optptr++ = DHCP_OPTION_END;
    return optptr;
}


////////////////////////////////////////////////////////////////////////////////
/// Initializes the DHCP message structure.
///
/// @param msg   Pointer to the DHCP message data.
/// @param netif Pointer to the network interface object.
///
/// @return Pointer to the optional parameters in the DHCP message structure.
////////////////////////////////////////////////////////////////////////////////
unsigned char *create_msg(pdhcp_msg msg, pnetif_data netif)
{
    //Clear the DHCP message structure.
    memset(msg, 0, sizeof(dhcp_msg));

    //Initialize the DHCP message structure.
    msg->op = DHCP_REQUEST;
    msg->htype = DHCP_HTYPE_ETHERNET;
    msg->hlen = sizeof(netif->macaddr);
    memcpy(msg->xid, xid, sizeof(xid));
    msg->flags = HTONS(BOOTP_BROADCAST);
    memcpy(msg->ciaddr, netif->ipaddr, sizeof(msg->ciaddr));
    memcpy(msg->chaddr, netif->macaddr, sizeof(netif->macaddr));
    memcpy(msg->options, magic_cookie, sizeof(magic_cookie));

    return &(msg->options[4]);
}


////////////////////////////////////////////////////////////////////////////////
/// Sends a DHCP discover message.
///
/// @param netif Pointer to the network interface object.
////////////////////////////////////////////////////////////////////////////////
void send_discover(pnetif_data netif)
{
    peth_header eth_hdr = (peth_header)(netif->packet);
    pip_header  ip_hdr  = (pip_header)(netif->packet + sizeof(eth_header));
    pudp_header udp_hdr = (pudp_header)(netif->packet + sizeof(eth_header) + sizeof(ip_header));
    pdhcp_msg   msg     = (pdhcp_msg)(netif->packet + sizeof(eth_header) + sizeof(ip_header) + sizeof(udp_header));
    unsigned char *end;
    unsigned short len;

    //Create the DHCP discover message.
    end = create_msg(msg, netif);
    end = add_msg_type(end, DHCPDISCOVER);
    end = add_req_options(end);
    end = add_end(end);

    len = sizeof(udp_header) + (end - (unsigned char *)msg);
    if (len < 308)
        len = 308;

    //Fill in the UDP header.
    udp_hdr->sourceport = HTONS(DHCPC_CLIENT_PORT);
    udp_hdr->destport = HTONS(DHCPC_SERVER_PORT);
    udp_hdr->length = HTONS(len);
    udp_hdr->checksum = 0;

    len += sizeof(ip_header);

    //Fill in the IP header.
    ip_hdr->ver_hlen = 0x45;
    ip_hdr->service = 0;
    ip_hdr->packetlength = HTONS(len);
    ip_hdr->ident = 0x55aa;
    ip_hdr->flags_offset = 0;
    ip_hdr->timetolive = 255;
    ip_hdr->protocol = IP_PROTOCOL_UDP;
    ip_hdr->checksum = 0;
    //Source IP: our current IP address.
    //memcpy(ip_hdr->sourceip, netif->ipaddr, 4);
    memset(ip_hdr->sourceip, 0, 4);
    //Destination IP: 255.255.255.255 (broadcast).
    memset(ip_hdr->destip, 255, 4);
    //memcpy(ip_hdr->destip, destip, 4);

    //Calculate the checksums.
    udp_hdr->checksum = udp_checksum(ip_hdr);
    ip_hdr->checksum = ip_checksum(ip_hdr);

    len += sizeof(eth_header);

    //Fill in the Ethernet header.
    //Destination MAC address: FF-FF-FF-FF-FF-FF (broadcast).
    memset(eth_hdr->destMAC, 0xff, 6);
    //Source MAC address: the MAC address of the network adapter.
    memcpy(eth_hdr->sourceMAC, netif->macaddr, 6);
    eth_hdr->length_type = HTONS(ETH_TYPE_IP);

    //Send the Ethernet packet.
    netif->txbytes = len;
    netif->netif_tx(netif);
    netif->txbytes = 0;
}


////////////////////////////////////////////////////////////////////////////////
/// Sends a DHCP request message.
///
/// @param netif Pointer to the network interface object.
////////////////////////////////////////////////////////////////////////////////
void send_request(pnetif_data netif)
{
    peth_header eth_hdr = (peth_header)netif->packet;
    pip_header  ip_hdr  = (pip_header)(netif->packet + sizeof(eth_header));
    pudp_header udp_hdr = (pudp_header)(netif->packet + sizeof(eth_header) + sizeof(ip_header));
    pdhcp_msg   msg     = (pdhcp_msg)(netif->packet + sizeof(eth_header) + sizeof(ip_header) + sizeof(udp_header));
    unsigned char *end;
    unsigned short len;

    //Create the DHCP request message.
    end = create_msg(msg, netif);
    end = add_msg_type(end, DHCPREQUEST);
    end = add_server_id(end, &netif->dhcp_state);
    end = add_req_ipaddr(end, &netif->dhcp_state);
    end = add_end(end);

    len = sizeof(udp_header) + (end - (unsigned char *)msg);
    if (len < 308)
        len = 308;

    //Fill in the UDP header.
    udp_hdr->sourceport = DHCPC_CLIENT_PORT;
    udp_hdr->destport = DHCPC_SERVER_PORT;
    udp_hdr->length = HTONS(len);
    udp_hdr->checksum = 0;

    len += sizeof(ip_header);

    //Fill in the IP header.
    ip_hdr->ver_hlen = 0x45;
    ip_hdr->service = 0;
    ip_hdr->packetlength = HTONS(len);
    ip_hdr->ident = 0;
    ip_hdr->flags_offset = 0;
    ip_hdr->timetolive = 255;
    ip_hdr->protocol = IP_PROTOCOL_UDP;
    ip_hdr->checksum = 0;
    //Source IP: our current IP address.
    memcpy(ip_hdr->sourceip, netif->ipaddr, 4);
    //Destination IP: the IP address of the DHCP server.
    memcpy(ip_hdr->destip, netif->dhcp_state.serverid, 4);

    //Calculate the checksums.
    udp_hdr->checksum = udp_checksum(ip_hdr);
    ip_hdr->checksum = ip_checksum(ip_hdr);

    len += sizeof(eth_header);

    //Fill in the Ethernet header.
    //Destination MAC address: the MAC address of the DHCP server.
    memcpy(eth_hdr->destMAC, netif->dhcp_state.servermac, 6);
    //Source MAC address: the MAC address of the network adapter.
    memcpy(eth_hdr->sourceMAC, netif->macaddr, 6);
    eth_hdr->length_type = HTONS(ETH_TYPE_IP);

    //Send the Ethernet packet.
    netif->txbytes = len;
    netif->netif_tx(netif);
    netif->txbytes = 0;
}


////////////////////////////////////////////////////////////////////////////////
/// Parses the options field of the received DHCP message.
///
/// @param optptr     Pointer to the options in the received DHCP message.
/// @param len        Size of the received DHCP options in bytes.
/// @param dhcp_state Pointer to the DHCP data object.
///
/// @return DHCP message type.
////////////////////////////////////////////////////////////////////////////////
unsigned char parse_options(unsigned char *optptr, unsigned long len, pdhcp_state dhcp_state)
{
    unsigned char *end = optptr + len;
    unsigned char type = 0;
    unsigned long lease_time;

    while(optptr < end)
    {
        switch(*optptr)
        {
            case DHCP_OPTION_SUBNET_MASK:
                memcpy(dhcp_state->netmask, optptr + 2, 4);
                break;
            case DHCP_OPTION_ROUTER:
                memcpy(dhcp_state->default_router, optptr + 2, 4);
                break;
            case DHCP_OPTION_DNS_SERVER:
                memcpy(dhcp_state->dnsaddr, optptr + 2, 4);
                break;
            case DHCP_OPTION_MSG_TYPE:
                type = *(optptr + 2);
                break;
            case DHCP_OPTION_SERVER_ID:
                memcpy(dhcp_state->serverid, optptr + 2, 4);
                break;
            case DHCP_OPTION_LEASE_TIME:
                memcpy(&lease_time, optptr + 2, 4);
                dhcp_state->lease_time = NTOHL(lease_time);
                break;
            case DHCP_OPTION_END:
                return type;
        }

        optptr += optptr[1] + 2;
    }

    return type;
}


////////////////////////////////////////////////////////////////////////////////
/// Parses the received DHCP message.
///
/// @param netif Pointer to the network interface object.
/// @param data  Pointer to the received data.
/// @param len   Size of the received DHCP options in bytes.
///
/// @return DHCP message type.
////////////////////////////////////////////////////////////////////////////////
unsigned char parse_msg(pnetif_data netif, void *data, unsigned long len)
{
    pdhcp_msg msg = (pdhcp_msg)data;

    if((msg->op == DHCP_REPLY) &&
       (memcmp(msg->xid, xid, sizeof(xid)) == 0) &&
       (memcmp(msg->chaddr, netif->macaddr, sizeof(netif->macaddr)) == 0))
    {
        memcpy(netif->dhcp_state.ipaddr, msg->yiaddr, 4);
        return parse_options(&msg->options[4], len, &netif->dhcp_state);
    }

    return 0;
}


////////////////////////////////////////////////////////////////////////////////
/// Initializes the DHCP client.
///
/// @param netif Pointer to the network interface object.
////////////////////////////////////////////////////////////////////////////////
void dhcp_init(pnetif_data netif)
{
    //Set our IP address to 0.0.0.0.
    //memset(netif->ipaddr, 0, 4);
    //Set the initial DHCP state.
    netif->dhcp_state.state = DHCP_STATE_DISCOVER;
    //Initialize the DHCP timer.
    netif->dhcp_state.ticks = CLOCK_SECOND;
    timer_set(&netif->dhcp_state.tmr, netif->dhcp_state.ticks);
    //Send a DHCP discover message.
    send_discover(netif);
}


/*
void print_dhcp_data(pnetif_data netif)
{
    xil_printf("Got IP address %d.%d.%d.%d\r\n",
               (unsigned long)(netif->dhcp_state.ipaddr[0]), (unsigned long)(netif->dhcp_state.ipaddr[1]),
               (unsigned long)(netif->dhcp_state.ipaddr[2]), (unsigned long)(netif->dhcp_state.ipaddr[3]));
    xil_printf("Got netmask %d.%d.%d.%d\r\n",
               (unsigned long)(netif->dhcp_state.netmask[0]), (unsigned long)(netif->dhcp_state.netmask[1]),
               (unsigned long)(netif->dhcp_state.netmask[2]), (unsigned long)(netif->dhcp_state.netmask[3]));
    xil_printf("Got DNS server %d.%d.%d.%d\r\n",
               (unsigned long)(netif->dhcp_state.dnsaddr[0]), (unsigned long)(netif->dhcp_state.dnsaddr[1]),
               (unsigned long)(netif->dhcp_state.dnsaddr[2]), (unsigned long)(netif->dhcp_state.dnsaddr[3]));
    xil_printf("Got default router %d.%d.%d.%d\r\n",
               (unsigned long)(netif->dhcp_state.default_router[0]), (unsigned long)(netif->dhcp_state.default_router[1]),
               (unsigned long)(netif->dhcp_state.default_router[2]), (unsigned long)(netif->dhcp_state.default_router[3]));
    xil_printf("Lease expires in %d seconds\r\n\r\n", netif->dhcp_state.lease_time);
}
*/


////////////////////////////////////////////////////////////////////////////////
/// DHCP client state machine. Processes the received DHCP data.
///
/// @param netif Pointer to the network interface object.
/// @param len   Size of the received data in bytes.
/// @param flags DHCP client event flags.
////////////////////////////////////////////////////////////////////////////////
void process_dhcp(pnetif_data netif, unsigned long len, unsigned long flags)
{
    peth_header eth_hdr = (peth_header)netif->packet;
    pdhcp_msg   msg     = (pdhcp_msg)(netif->packet + sizeof(eth_header) + sizeof(ip_header) + sizeof(udp_header));

    //Check the current state.
    switch (netif->dhcp_state.state)
    {
        case DHCP_STATE_DISCOVER:
            //Check whether data has been received.
            if ((flags & DHCP_FLAG_NEW_DATA) && (parse_msg(netif, msg, len) == DHCPOFFER))
            {
                //Save the MAC address of the DHCP server.
                memcpy(netif->dhcp_state.servermac, eth_hdr->sourceMAC, 6);
                //Set the next state.
                netif->dhcp_state.state = DHCP_STATE_OFFER_RECEIVED;
                //Set the timer.
                netif->dhcp_state.ticks = CLOCK_SECOND;
                timer_set(&netif->dhcp_state.tmr, netif->dhcp_state.ticks);
                //Send a DHCP request message.
                send_request(netif);
                break;
            }
            //Check whether the timer has been expired.
            if (flags & DHCP_FLAG_TIMER)
            {
                //Set the timer.
                if (netif->dhcp_state.ticks < (CLOCK_SECOND * 60))
                {
                    //netif->dhcp_state.ticks *= 2;
                    netif->dhcp_state.ticks = CLOCK_SECOND * 2;
                    timer_set(&netif->dhcp_state.tmr, netif->dhcp_state.ticks);
                }
                //Send a DHCP discover message.
                send_discover(netif);
            }
            break;
        case DHCP_STATE_OFFER_RECEIVED:
            //Check whether data has been received.
            if ((flags & DHCP_FLAG_NEW_DATA) && (parse_msg(netif, msg, len) == DHCPACK))
            {
                //Set our new IP address.
                memcpy(netif->ipaddr, netif->dhcp_state.ipaddr, 4);
                //Set the next state.
                netif->dhcp_state.state = DHCP_STATE_CONFIG_RECEIVED;
                //print_dhcp_data(netif);
                //Set the timer.
                netif->dhcp_state.ticks = CLOCK_SECOND;
                timer_set(&netif->dhcp_state.tmr, netif->dhcp_state.ticks);
                break;
            }
            //Check whether the timer has been expired.
            if (flags & DHCP_FLAG_TIMER)
            {
                //Check the timer interval.
                if (netif->dhcp_state.ticks <= (CLOCK_SECOND * 10))
                {
                    //Send a DHCP request message and increment the timer interval.
                    send_request(netif);
                    netif->dhcp_state.ticks += CLOCK_SECOND;
                }
                else
                {
                    //Restart.
                    send_discover(netif);
                    netif->dhcp_state.state = DHCP_STATE_DISCOVER;
                    netif->dhcp_state.ticks = CLOCK_SECOND;
                }
                //Set the timer.
                timer_set(&netif->dhcp_state.tmr, netif->dhcp_state.ticks);
            }
            break;
        case DHCP_STATE_CONFIG_RECEIVED:
            //Check whether the timer has been expired.
            if ((flags & DHCP_FLAG_TIMER))
            {
                //Decrease the lease time.
                netif->dhcp_state.lease_time--;
                //Get a new IP address if the lease time has been elapsed.
                if (netif->dhcp_state.lease_time == 0)
                {
                    //Invalidate our IP address.
                    memset(netif->ipaddr, 0, 4);
                    //Set the next state.
                    netif->dhcp_state.state = DHCP_STATE_DISCOVER;
                    //Send a DHCP discover message.
                    send_discover(netif);
                }
            }
            break;
    }
}

