////////////////////////////////////////////////////////////////////////////////
/// @file netif.c
/// Network interface handler functions.
///
/// @author  Tamas Raikovich
/// @version 1.0
/// @date    2012.01.21.
////////////////////////////////////////////////////////////////////////////////
#include "netif.h"
#include "udp.h"
#include "dhcp_client.h"


///Pointer to the first network interface object in the list.
pnetif_data first_netif;
///Pointer to the last network interface object in the list.
pnetif_data last_netif;


////////////////////////////////////////////////////////////////////////////////
/// Initializes the network interface handler.
////////////////////////////////////////////////////////////////////////////////
void netif_init()
{
    first_netif = (pnetif_data)0;
    last_netif = (pnetif_data)0;
}


////////////////////////////////////////////////////////////////////////////////
/// Adds a network interface to the system.
///
/// @param netif      Pointer to the network interface object.
/// @param baseaddr   Base address of the network adapter.
/// @param macaddr    Pointer to the MAC address of the network adapter.
/// @param ipaddr     Pointer to the IP address of the network adapter.
/// @param use_dhcp   Nonzero if use DHCP to get the IP address.
/// @param netif_init Pointer to the network adapter initialization function.
/// @param netif_rx   Pointer to the network adapter packet RX function.
/// @param netif_tx   Pointer to the network adapter packet TX function.
////////////////////////////////////////////////////////////////////////////////
void netif_add(
    pnetif_data   netif,
    unsigned long baseaddr,
    unsigned char *macaddr,
    unsigned char *ipaddr,
    unsigned char use_dhcp,
    unsigned long (*netif_init)(pnetif_data),
    unsigned long (*netif_rx)(pnetif_data),
    unsigned long (*netif_tx)(pnetif_data)
)
{
    //Store the base address of the network adapter.
    netif->baseaddr = baseaddr;

    //Store the MAC address of the network adapter.
    if (macaddr != 0)
    {
        netif->macaddr[0] = macaddr[0];
        netif->macaddr[1] = macaddr[1];
        netif->macaddr[2] = macaddr[2];
        netif->macaddr[3] = macaddr[3];
        netif->macaddr[4] = macaddr[4];
        netif->macaddr[5] = macaddr[5];
    }

    //Store the IP address of the network adapter.
    if (ipaddr != 0)
    {
        netif->ipaddr[0] = ipaddr[0];
        netif->ipaddr[1] = ipaddr[1];
        netif->ipaddr[2] = ipaddr[2];
        netif->ipaddr[3] = ipaddr[3];
    }

    netif->use_dhcp = use_dhcp;

    //Store the function pointers.
    netif->netif_init = netif_init;
    netif->netif_rx = netif_rx;
    netif->netif_tx = netif_tx;

    //Add the network interface to the list.
    if (first_netif == (pnetif_data)0)
    {
        first_netif = netif;
    }
    else
    {
        last_netif->next = netif;
    }

    last_netif = netif;
    netif->next = (pnetif_data)0;
}


////////////////////////////////////////////////////////////////////////////////
/// Initializes the network adapters.
////////////////////////////////////////////////////////////////////////////////
void netif_start()
{
    pnetif_data netif = first_netif;

    //Call the initialization function of each network adapter.
    while (netif != (pnetif_data)0)
    {
        netif->netif_init(netif);
        if (netif->use_dhcp)
            dhcp_init(netif);
        netif = netif->next;
    }
}


////////////////////////////////////////////////////////////////////////////////
/// Processes the received packets and sends the available packets for all
/// network adapters.
////////////////////////////////////////////////////////////////////////////////
void netif_process_all()
{
    pnetif_data netif = first_netif;

    //Go through the list of the network adapters.
    while (netif != (pnetif_data)0)
    {
    	netif_process(netif);
        netif = netif->next;
    }
}


////////////////////////////////////////////////////////////////////////////////
/// Processes the received packets and sends the available packets for the
/// given network adapter.
///
/// @param netif Pointer to the network interface object.
////////////////////////////////////////////////////////////////////////////////
void netif_process(pnetif_data netif)
{
    //Check whether the DHCP timer has expired.
    if ((netif->use_dhcp) && timer_expired(&netif->dhcp_state.tmr))
    {
        timer_reset(&netif->dhcp_state.tmr);
        process_dhcp(netif, 0, DHCP_FLAG_TIMER);
    }

    //Check whether a packet has been received.
    netif->netif_rx(netif);

    if (netif->rxbytes > 0)
    {
    	//Process the received packet.
    	netif->txbytes = 0;
    	process_ethernet(netif);

    	//Send the packet if necessary.
    	if (netif->txbytes > 0)
    		netif->netif_tx(netif);
    }
}

