////////////////////////////////////////////////////////////////////////////////
/// @file netif.h
/// Header file for the network interface handler functions.
///
/// @author  Tamas Raikovich
/// @version 1.0
/// @date    2012.01.21.
////////////////////////////////////////////////////////////////////////////////
#ifndef NETIF_H_
#define NETIF_H_

#include "dhcp_client.h"


////////////////////////////////////////////////////////////////////////////////
/// Maximum size of the network packet in bytes that can be received.
////////////////////////////////////////////////////////////////////////////////
#define NETIF_MAX_PACKET_SIZE       1520


////////////////////////////////////////////////////////////////////////////////
/// Set to 1 if the host is big-endian. Set to 0 if the host is little-endian.
////////////////////////////////////////////////////////////////////////////////
#define HOST_IS_BIG_ENDIAN          1


////////////////////////////////////////////////////////////////////////////////
/// Network interface object.
////////////////////////////////////////////////////////////////////////////////
typedef struct _netif_data
{
	///Next RX packet pointer.
	unsigned long nextRxPacketPtr;
    ///Base address of the network interface.
    unsigned long baseaddr;
    ///MAC address of the network interface.
    unsigned char macaddr[6];
    ///IP address of the network interface.
    unsigned char ipaddr[4];
    ///Reserved.
    unsigned char reserved[2];
    ///Packet buffer.
    unsigned char packet[NETIF_MAX_PACKET_SIZE];
    ///Pointer to the user application data in the packet buffer.
    unsigned char *userdata;
    ///Number of received bytes.
    unsigned long rxbytes;
    ///Number of bytes to send.
    unsigned long txbytes;
    ///Pointer to the network interface initialization function.
    unsigned long (*netif_init)(struct _netif_data *);
    ///Pointer to the data RX function.
    unsigned long (*netif_rx)(struct _netif_data *);
    ///Pointer to the data TX function.
    unsigned long (*netif_tx)(struct _netif_data *);
    ///Indicates whether DHCP is used for this network interface.
    unsigned char use_dhcp;
    ///DHCP data object.
    struct _dhcp_state dhcp_state;
    ///Pointer to the next network interface.
    struct _netif_data *next;
} netif_data, *pnetif_data;


////////////////////////////////////////////////////////////////////////////////
///@name Host to network and network to host byte order conversion macros.
///@{
////////////////////////////////////////////////////////////////////////////////
#if (HOST_IS_BIG_ENDIAN != 0)

///Converts a 16-bit value from host to network byte order.
#define HTONS(a) (a)
///Converts a 32-bit value from host to network byte order.
#define HTONL(a) (a)

#else

///Converts a 16-bit value from host to network byte order.
#define HTONS(a) ((unsigned short)(((unsigned short)(a) <<8 ) | ((unsigned short)(a) >> 8)))

///Converts a 32-bit value from host to network byte order.
#define HTONL(a) ((unsigned long)((((unsigned long)(a) & 0x000000ff) << 24) | \
                                  (((unsigned long)(a) & 0x0000ff00) << 8)  | \
                                  (((unsigned long)(a) & 0x00ff0000) >> 8)  | \
                                  (((unsigned long)(a) & 0xff000000) >> 24)))

#endif

///Converts a 16-bit value from network to host byte order.
#define NTOHS(a) HTONS(a)
///Converts a 32-bit value from network to host byte order.
#define NTOHL(a) HTONL(a)
///@}


////////////////////////////////////////////////////////////////////////////////
///@name Function prototypes.
///@{
////////////////////////////////////////////////////////////////////////////////
void netif_init();

void netif_add(
    pnetif_data netif,                          //Pointer to the network interface structure.
    unsigned long baseaddr,                     //Base address of the network adapter.
    unsigned char *macaddr,                     //Pointer to the MAC address of the network adapter.
    unsigned char *ipaddr,                      //Pointer to the IP address of the network adapter.
    unsigned char use_dhcp,                     //Use DHCP to get the IP address.
    unsigned long (*netif_init)(pnetif_data),   //Pointer to the network adapter initialization function.
    unsigned long (*netif_rx)(pnetif_data),     //Pointer to the packet RX function.
    unsigned long (*netif_tx)(pnetif_data)      //Pointer to the packet TX function.
);

void netif_start();

void netif_process_all();
void netif_process(pnetif_data netif);

void dhcp_init(pnetif_data netif);
void process_dhcp(pnetif_data netif, unsigned long len, unsigned long flags);
///@}


#endif /* NETIF_H_ */
