//******************************************************************************
//* Xilinx CacheLink interface module.                                         *
//*                                                                            *
//* Written by   : Tamas Raikovich                                             *
//* Version      : 1.0                                                         *
//* Last modified: 2012.10.21.                                                 *
//******************************************************************************
module xcl_interface #(
   //XCL0 interface parameters.
   parameter XCL0_CACHELINE_SIZE = 4,        //Cacheline size in 32-bit words
   parameter XCL0_IN_FIFO_DEPTH  = 16,       //Depth of the XCL input FIFO
   parameter XCL0_OUT_FIFO_DEPTH = 16,       //Depth of the XCL output FIFO
   
   //XCL1 interface parameters.
   parameter XCL1_CACHELINE_SIZE = 4,        //Cacheline size in 32-bit words
   parameter XCL1_IN_FIFO_DEPTH  = 16,       //Depth of the XCL input FIFO
   parameter XCL1_OUT_FIFO_DEPTH = 16,       //Depth of the XCL output FIFO
  
   //Memory read pipeline delay in clocks.
   parameter MEM_RD_DELAY_CLK  = 4
) (
   //Clock and reset.
   input  wire        clk,                   //Clock signal
   input  wire        rst,                   //Reset signal
   
   //CacheLink interface 0 signals (big-endian).
   input  wire        xcl0_access_ctrl,      //XCL control input
   input  wire [0:31] xcl0_access_data,      //XCL input data
   input  wire        xcl0_access_wr,        //XCL input FIFO write signal
   output wire        xcl0_access_full,      //XCL input FIFO is full
   output wire        xcl0_rddata_ctrl,      //XCL control output
   output wire [0:31] xcl0_rddata_data,      //XCL data output
   input  wire        xcl0_rddata_rd,        //XCL output FIFO read signal
   output wire        xcl0_rddata_exists,    //XCL output FIFO is not empty
   
   //CacheLink interface 1 signals (big-endian).
   input  wire        xcl1_access_ctrl,      //XCL control input
   input  wire [0:31] xcl1_access_data,      //XCL input data
   input  wire        xcl1_access_wr,        //XCL input FIFO write signal
   output wire        xcl1_access_full,      //XCL input FIFO is full
   output wire        xcl1_rddata_ctrl,      //XCL control output
   output wire [0:31] xcl1_rddata_data,      //XCL data output
   input  wire        xcl1_rddata_rd,        //XCL output FIFO read signal
   output wire        xcl1_rddata_exists,    //XCL output FIFO is not empty
   
   //Memory controller interface signals (little-endian).
   output wire        mem_write_req,         //Write transfer request signal
   output wire        mem_read_req,          //Read transfer request signal
   output wire [24:1] mem_address,           //Address output
   output reg  [3:0]  mem_byte_en,           //Byte enable signals
   input  wire        mem_write_ack,         //Write acknowledge signal
   output wire        mem_wr_valid,          //Write data valid signal
   output wire [31:0] mem_wr_data,           //Write data output
   input  wire        mem_read_ack,          //Read acknowledge signal
   input  wire        mem_rd_valid,          //Read data valid signal
   input  wire [31:0] mem_rd_data            //Read data input
);

//******************************************************************************
//* XCL0 input FIFO.                                                           *
//******************************************************************************
wire [32:0] in0_fifo_din = {xcl0_access_ctrl, xcl0_access_data};
wire [32:0] in0_fifo_dout;
wire        in0_fifo_rd;
wire        in0_fifo_exists;

fifo #(
   //Depth of the FIFO.
   .DEPTH(XCL0_IN_FIFO_DEPTH),
   //Number of bits in the data words.
   .WIDTH(33),
   //The programmable full flag is set to 1 when the FIFO
   //contains at least PRG_FULL_H_TRESH data words.
   .PRG_FULL_H_TRESH(XCL0_IN_FIFO_DEPTH / 2),
   //The programmable full flag is set to 0 when the FIFO
   //contains less than PRG_FULL_L_TRESH data words.
   .PRG_FULL_L_TRESH(XCL0_IN_FIFO_DEPTH / 2)
) xcl_in0_fifo (
   //Clock and reset.
   .clk(clk),                                //Clock signal
   .rst(rst),                                //Reset signal
   
   //Data input and output.
   .data_in(in0_fifo_din),                   //FIFO data input
   .data_out(in0_fifo_dout),                 //FIFO data output
   
   //Control inputs.
   .write(xcl0_access_wr),                   //FIFO write signal
   .read(in0_fifo_rd),                       //FIFO read signal
   
   //Status outputs.
   .exists(in0_fifo_exists),                 //FIFO is not empty
   .full(xcl0_access_full),                  //FIFO is full
   .prg_full()                               //Programmable FULL flag
);


//******************************************************************************
//* XCL1 input FIFO.                                                           *
//******************************************************************************
wire [32:0] in1_fifo_din = {xcl1_access_ctrl, xcl1_access_data};
wire [32:0] in1_fifo_dout;
wire        in1_fifo_rd;
wire        in1_fifo_exists;

fifo #(
   //Depth of the FIFO.
   .DEPTH(XCL1_IN_FIFO_DEPTH),
   //Number of bits in the data words.
   .WIDTH(33),
   //The programmable full flag is set to 1 when the FIFO
   //contains at least PRG_FULL_H_TRESH data words.
   .PRG_FULL_H_TRESH(XCL1_IN_FIFO_DEPTH / 2),
   //The programmable full flag is set to 0 when the FIFO
   //contains less than PRG_FULL_L_TRESH data words.
   .PRG_FULL_L_TRESH(XCL1_IN_FIFO_DEPTH / 2)
) xcl_in1_fifo (
   //Clock and reset.
   .clk(clk),                                //Clock signal
   .rst(rst),                                //Reset signal
   
   //Data input and output.
   .data_in(in1_fifo_din),                   //FIFO data input
   .data_out(in1_fifo_dout),                 //FIFO data output
   
   //Control inputs.
   .write(xcl1_access_wr),                   //FIFO write signal
   .read(in1_fifo_rd),                       //FIFO read signal
   
   //Status outputs.
   .exists(in1_fifo_exists),                 //FIFO is not empty
   .full(xcl1_access_full),                  //FIFO is full
   .prg_full()                               //Programmable FULL flag
);


//******************************************************************************
//* Arbiter for the XCL0 and XCL1 channels. Least recently used arbitration    *
//* scheme is used.                                                            *
//******************************************************************************
reg        xcl_sel_reg;
wire       xcl_sel_reg_ld;
reg  [1:0] arbiter_rom_data;
wire [2:0] arbiter_rom_addr = {xcl_sel_reg, in1_fifo_exists, in0_fifo_exists};

//Arbiter ROM.
always @(*)
begin
   case (arbiter_rom_addr)
      //Selected XCL channel   --------\
      //A request is available ------\ |
      //                             | |
      //                             V V
      3'b000: arbiter_rom_data <= 2'b0_0;    //No request
      3'b001: arbiter_rom_data <= 2'b1_0;    //XCL0 is selected
      3'b010: arbiter_rom_data <= 2'b1_1;    //XCL1 is selected
      3'b011: arbiter_rom_data <= 2'b1_1;    //XCL1 is selected
      3'b100: arbiter_rom_data <= 2'b0_0;    //No request
      3'b101: arbiter_rom_data <= 2'b1_0;    //XCL0 is selected
      3'b110: arbiter_rom_data <= 2'b1_1;    //XCL1 is selected
      3'b111: arbiter_rom_data <= 2'b1_0;    //XCL0 is selected
   endcase
end

//Request available signal.
wire req_available = arbiter_rom_data[1];

//XCL channel select signal.
//0: XCL0 is selected, XCL1 has the priority at the next arbitration.
//1: XCL1 is selected, XCL0 has the priority at the next arbitration.
always @(posedge clk)
begin
   if (rst)
      xcl_sel_reg <= 1'b0;
   else
      if (xcl_sel_reg_ld)
         xcl_sel_reg <= arbiter_rom_data[0];
end

//Data exists in the selected input FIFO.
wire in_fifo_exists = (xcl_sel_reg) ? in1_fifo_exists : in0_fifo_exists;

//Input FIFO select multiplexers.
wire [32:0] xcl_addr_mux = (arbiter_rom_data[0]) ? in1_fifo_dout : in0_fifo_dout;
wire [32:0] xcl_data_mux = (xcl_sel_reg)         ? in1_fifo_dout : in0_fifo_dout;

//Driving the memory write data output (big-endian to little-endian conversion).
assign mem_wr_data[7:0]   = xcl_data_mux[31:24];
assign mem_wr_data[15:8]  = xcl_data_mux[23:16];
assign mem_wr_data[23:16] = xcl_data_mux[15:8];
assign mem_wr_data[31:24] = xcl_data_mux[7:0];

//Delaying the XCL channel select signal.
reg [MEM_RD_DELAY_CLK-1:0] xcl_sel_delay_reg;

always @(posedge clk)
begin
   xcl_sel_delay_reg <= {xcl_sel_reg, xcl_sel_delay_reg[MEM_RD_DELAY_CLK-1:1]};
end

wire xcl_sel_delayed = xcl_sel_delay_reg[0];


//******************************************************************************
//* XCL0 output FIFO.                                                          *
//******************************************************************************
wire [31:0] out_fifo_din;
wire        out0_fifo_full;
wire        out0_fifo_wr = mem_rd_valid & ~xcl_sel_delayed;

//Little-endian to big-endian conversion.
assign out_fifo_din[31:24] = mem_rd_data[7:0];
assign out_fifo_din[23:16] = mem_rd_data[15:8];
assign out_fifo_din[15:8]  = mem_rd_data[23:16];
assign out_fifo_din[7:0]   = mem_rd_data[31:24];

fifo #(
   //Depth of the FIFO.
   .DEPTH(XCL0_OUT_FIFO_DEPTH),
   //Number of bits in the data words.
   .WIDTH(32),
   //The programmable full flag is set to 1 when the FIFO
   //contains at least PRG_FULL_H_TRESH data words.
   .PRG_FULL_H_TRESH(XCL0_OUT_FIFO_DEPTH - MEM_RD_DELAY_CLK - 1),
   //The programmable full flag is set to 0 when the FIFO
   //contains less than PRG_FULL_L_TRESH data words.
   .PRG_FULL_L_TRESH(MEM_RD_DELAY_CLK + 3)
) xcl_out0_fifo (
   //Clock and reset.
   .clk(clk),                                //Clock signal
   .rst(rst),                                //Reset signal
   
   //Data input and output.
   .data_in(out_fifo_din),                   //FIFO data input
   .data_out(xcl0_rddata_data),              //FIFO data output
   
   //Control inputs.
   .write(out0_fifo_wr),                     //FIFO write signal
   .read(xcl0_rddata_rd),                    //FIFO read signal
   
   //Status outputs.
   .exists(xcl0_rddata_exists),              //FIFO is not empty
   .full(),                                  //FIFO is full
   .prg_full(out0_fifo_full)                 //Programmable FULL flag
);

assign xcl0_rddata_ctrl = 1'b0;


//******************************************************************************
//* XCL1 output FIFO.                                                          *
//******************************************************************************
wire out1_fifo_full;
wire out1_fifo_wr = mem_rd_valid & xcl_sel_delayed;

fifo #(
   //Depth of the FIFO.
   .DEPTH(XCL1_OUT_FIFO_DEPTH),
   //Number of bits in the data words.
   .WIDTH(32),
   //The programmable full flag is set to 1 when the FIFO
   //contains at least PRG_FULL_H_TRESH data words.
   .PRG_FULL_H_TRESH(XCL1_OUT_FIFO_DEPTH - MEM_RD_DELAY_CLK - 1),
   //The programmable full flag is set to 0 when the FIFO
   //contains less than PRG_FULL_L_TRESH data words.
   .PRG_FULL_L_TRESH(MEM_RD_DELAY_CLK + 3)
) xcl_out1_fifo (
   //Clock and reset.
   .clk(clk),                                //Clock signal
   .rst(rst),                                //Reset signal
   
   //Data input and output.
   .data_in(out_fifo_din),                   //FIFO data input
   .data_out(xcl1_rddata_data),              //FIFO data output
   
   //Control inputs.
   .write(out1_fifo_wr),                     //FIFO write signal
   .read(xcl1_rddata_rd),                    //FIFO read signal
   
   //Status outputs.
   .exists(xcl1_rddata_exists),              //FIFO is not empty
   .full(),                                  //FIFO is full
   .prg_full(out1_fifo_full)                 //Programmable FULL flag
);

assign xcl1_rddata_ctrl = 1'b0;

//FIFO full signal.
wire out_fifo_full = (xcl_sel_reg) ? out1_fifo_full : out0_fifo_full;


//******************************************************************************
//* Transfer direction register.                                               *
//******************************************************************************
reg  read_transfer;
wire addr_reg_ld;

always @(posedge clk)
begin
   if (addr_reg_ld)
      read_transfer <= ~xcl_addr_mux[32];
end


//******************************************************************************
//* Address counter.                                                           *
//******************************************************************************
reg  [24:0] addr_reg;
reg  [3:0]  addr_cnt_mask;
wire        addr_cnt_en;
wire [5:2]  next_address = addr_reg[5:2] + 4'd1;

integer i;

always @(posedge clk)
begin
   //addr_reg[1:0]: byte enable select.
   if (addr_reg_ld)
      addr_reg[1:0] <= xcl_addr_mux[1:0];
      
   //addr_reg[5:2]: address counter.
   for (i = 0; i < 4; i = i + 1)
      if (addr_reg_ld)
         addr_reg[i+2] <= xcl_addr_mux[i+2];
      else
         if (addr_cnt_en && addr_cnt_mask[i])
            addr_reg[i+2] <= next_address[i+2];
       
   //addr_reg[24:6]: upper address bits.
   if (addr_reg_ld)
      addr_reg[24:6] <= xcl_addr_mux[24:6];
end

//Driving the memory address output.
assign mem_address = {addr_reg[24:2], 1'b0};


//******************************************************************************
//* Byte enable register.                                                      *
//******************************************************************************
wire       be_reg_ld;
wire [2:0] be_select = {addr_reg[1:0], xcl_data_mux[32]};

always @(posedge clk)
begin
   if (be_reg_ld)
      if (read_transfer)
         mem_byte_en <= 4'b1111;                //Read (always burst)
      else
         case (be_select)
            3'b000: mem_byte_en <= 4'b1111;     //Write (word)
            3'b001: mem_byte_en <= 4'b0001;     //Write (byte 0)
            3'b010: mem_byte_en <= 4'b0011;     //Write (halfword 0)
            3'b011: mem_byte_en <= 4'b0010;     //Write (byte 1)
            3'b100: mem_byte_en <= 4'b1111;     //Write (burst)
            3'b101: mem_byte_en <= 4'b0100;     //Write (byte 2)
            3'b110: mem_byte_en <= 4'b1100;     //Write (halfword 1)
            3'b111: mem_byte_en <= 4'b1000;     //Write (byte 3)
         endcase
end


//******************************************************************************
//* Burst length counter.                                                      *
//******************************************************************************
reg  [3:0] burst_len_cnt;
wire       burst_len_cnt_en;
wire [1:0] burst_len_sel;

assign burst_len_sel[0] = xcl_sel_reg;
assign burst_len_sel[1] = read_transfer | (be_select == 3'b100);

always @(posedge clk)
begin
   if (be_reg_ld)
      case (burst_len_sel)
         2'b00: burst_len_cnt <= 4'b0000;
         2'b01: burst_len_cnt <= 4'b0000;
         2'b10: burst_len_cnt <= XCL0_CACHELINE_SIZE - 1;
         2'b11: burst_len_cnt <= XCL1_CACHELINE_SIZE - 1;
      endcase
   else
      if (burst_len_cnt_en)
         burst_len_cnt <= burst_len_cnt - 4'd1;
end

always @(posedge clk)
begin
   if (be_reg_ld)
      case (burst_len_sel)
         2'b00: addr_cnt_mask <= 4'b0000;
         2'b01: addr_cnt_mask <= 4'b0000;
         2'b10: addr_cnt_mask <= XCL0_CACHELINE_SIZE - 1;
         2'b11: addr_cnt_mask <= XCL1_CACHELINE_SIZE - 1;
      endcase
end

//Indicates the last word of the transfer.
wire last_word = (burst_len_cnt == 4'd0);


//******************************************************************************
//* Controller state machine.                                                  *
//******************************************************************************
localparam STATE_REQ_WAIT  = 3'd0;
localparam STATE_INIT      = 3'd1;
localparam STATE_MEM_WRITE = 3'd2;
localparam STATE_MEM_READ  = 3'd3;
localparam STATE_FIFO_FULL = 3'd4;

reg [2:0] xcl_state;

always @(posedge clk)
begin
   if (rst)
      xcl_state <= STATE_REQ_WAIT;
   else
      case (xcl_state)
         //Wait for the request. Load the transfer direction and
         //the address registers if a request is available.
         STATE_REQ_WAIT : if (req_available)
                             xcl_state <= STATE_INIT;
                          else
                             xcl_state <= STATE_REQ_WAIT;

         //Load the byte enable register and the burst length counter.
         STATE_INIT     : if (read_transfer)
                             if (out_fifo_full)
                                xcl_state <= STATE_FIFO_FULL;
                             else
                                xcl_state <= STATE_MEM_READ;
                          else
                             if (in_fifo_exists)
                                xcl_state <= STATE_MEM_WRITE;
                             else
                                xcl_state <= STATE_INIT;
                             
         //Servicing a write request.
         STATE_MEM_WRITE: if (mem_write_ack && in_fifo_exists && last_word)
                             xcl_state <= STATE_REQ_WAIT;
                          else
                             xcl_state <= STATE_MEM_WRITE;
         
         //Servicing a read request.
         STATE_MEM_READ : if (mem_read_ack)
                             if (last_word)
                                xcl_state <= STATE_REQ_WAIT;
                             else
                                if (out_fifo_full)
                                   xcl_state <= STATE_FIFO_FULL;
                                else
                                   xcl_state <= STATE_MEM_READ;
                          else
                             xcl_state <= STATE_MEM_READ;
         
         STATE_FIFO_FULL: if (out_fifo_full)
                             xcl_state <= STATE_FIFO_FULL;
                          else
                             xcl_state <= STATE_MEM_READ;
         
         //Invalid states.
         default        : xcl_state <= STATE_REQ_WAIT;
      endcase
end


//******************************************************************************
//* Driving the memory controller interface signals.                           *
//******************************************************************************
//Write transfer request signal.
assign mem_write_req = (xcl_state == STATE_MEM_WRITE) & ~(mem_write_ack & in_fifo_exists & last_word);
//Read transfer request signal.     
assign mem_read_req  = (xcl_state == STATE_MEM_READ)  & ~(mem_read_ack & (out_fifo_full | last_word));
//Write data valid signal
assign mem_wr_valid  = (xcl_state == STATE_MEM_WRITE) & in_fifo_exists;


//******************************************************************************
//* Register control signals.                                                  *
//******************************************************************************
//XCL interface select register load signal.
assign xcl_sel_reg_ld = (xcl_state == STATE_REQ_WAIT) & req_available;

//Address register control signals.
assign addr_reg_ld = (xcl_state == STATE_REQ_WAIT) & req_available;
assign addr_cnt_en = mem_write_ack | mem_read_ack;

//Byte enable register and burst length counter load signal.
assign be_reg_ld = (xcl_state == STATE_INIT) & (in_fifo_exists | read_transfer);

//Burst length counter enable signal.
assign burst_len_cnt_en = mem_write_ack | mem_read_ack;


//******************************************************************************
//* FIFO control signals.                                                      *
//******************************************************************************
assign in0_fifo_rd = (addr_reg_ld & ~arbiter_rom_data[0]) | (mem_write_ack & ~xcl_sel_reg);
assign in1_fifo_rd = (addr_reg_ld &  arbiter_rom_data[0]) | (mem_write_ack &  xcl_sel_reg);


endmodule
