function stop = check_stopcrit(history, stopcrit)
% -------------------------------------------------
% CLASSICAL GENETIC ALGORITHM
% check_stopcrit - check given stop-criteria of
%                  the genetic algorithm
%
% Made by:
% Daniel L. Kovacs
% <dkovacs@mit.bme.hu>
%
% Department of Measurement and Information Systems
% Faculty of Electrical Engineering and Informatics
% Budapest University of Technology and Economics
% 
% January 2010
% -------------------------------------------------

try
    
    % Initialize output variables
    stop = false;

	% -----------------------------------------------------------------------------------------------------------------------------
    % -------------------------------------------------------- INPUT CHECK --------------------------------------------------------
    % -----------------------------------------------------------------------------------------------------------------------------
    
    % Check input variable 1/2
	if  isempty(history) ||...                                          % IF the whole history is empty, OR...
        ~isfield(history, 'bestf') ||...                                % it hasn't got a "bestf" field, OR...
        ~isfield(history, 'best') ||...                                 % it hasn't got a "best" field, OR...
        ~isfield(history, 'avgf') ||...                                 % it hasn't got a "avgf" field, OR...
        numel(history.bestf) == 0 ||...                                 % the history of best fitnesses is empty, OR...
        numel(history.best) == 0 ||...                                  % the history of best individuals is empty, OR...
        numel(history.avgf) == 0,                                       % the history of average fitnesses is empty, THEN...
       
        throw(MException('MATLAB:check_stopcrit:fvals',...
                         'Input error (1/2): the history (and all of its elements) should be non-empty (at least the stats of the initial population (gen0) should be there)'));
        
	end
    
	% Check input variable 2/2
    scerr = false;
    
    if  ~(size(stopcrit, 1) == 1) ||...                                 % IF stop criteria (stopcrit) aren't given in a row vector, OR...
        ~(size(stopcrit, 2) >= 1),                                      % the row vector is empty, THEN...
    
        scerr = true;
    
    else

        % There should be maximally 1 stop-criterion of every type
        scnum.mingennum	= 0;
        scnum.minbestf	= 0;
        scnum.minavgf	= 0;
        
        for i=1:size(stopcrit, 2),
        
            if ~isstruct(stopcrit(i)) ||...                             % IF the i-th stop criterion (stopcrit(i)) isn't a structure, OR...
               ~isfield(stopcrit(i), 'ctype') ||...                     % it hasn't got a "ctype" field, OR...
               (~strcmp(stopcrit(i).ctype, 'mingennum') &&...           % its "ctype" field...
                ~strcmp(stopcrit(i).ctype, 'minbestf') &&...            % has a value...
                ~strcmp(stopcrit(i).ctype, 'minavgf')) ||...            % different from 'mingennum' OR 'minbestf' OR 'minavgf', OR...
               scnum.(stopcrit(i).ctype) ~= 0 ||...                     % there was already a j-th (j < i) stop criterion with the same "ctype", OR...
               ~isfield(stopcrit(i), 'cparams') ||...                   % the i-th stop criterion hasn't got a "cparams" field, OR...
               isempty(stopcrit(i).cparams),                            % or that "cparams" field is empty, THEN...
           
                scerr = true;
                break;
           
            end
            
            % Increment the number of the criteria having the type of
            % the i-th stop criterion among the stop criteria
            scnum.(stopcrit(i).ctype) = scnum.(stopcrit(i).ctype) + 1;
            
        end
        
    end
    
    % IF there was some error concerning given stop criteria, THEN...
    if scerr,
        
        throw(MException('MATLAB:check_stopcrit:stopcrit',...
                         'Input error (2/2): the stop-criteria should be given as an 1*N array (N >= 1) with structures as elements having distinct, non-empty "ctype" string-fields (with a value being either "mingennum", or "minbestf", or "minavgf") and arbitrary, but non-empty "cparams" fields'));
        
    end
    
    % --------------------------------------------------------------------------------------------------------------------------------
    % -------------------------------------------------------- FUNCTION BODY  --------------------------------------------------------
    % --------------------------------------------------------------------------------------------------------------------------------
    
    i = 0;
    
    % Let's check all the stop criteria
    while (stop == false && i < size(stopcrit, 2)),
       
        i = i + 1;

        switch stopcrit(i).ctype
            
            case 'mingennum'
               
                if history.gen >= stopcrit(i).cparams,
                   
                    stop = true;
                    
                end
              
            case 'minbestf'
               
                if history.bestf(size(history.bestf, 2)) >= stopcrit(i).cparams,
                   
                    stop = true;
                    
                end
              
            case 'minavgf'
               
                if history.avgf(size(history.avgf, 2)) >= stopcrit(i).cparams,
                   
                    stop = true;
                    
                end
              
            otherwise
               
                throw(MException('MATLAB:check_stopcrit:stopcrit_i',...
                                 'Input error (2/2): unknown stop-criterion type'));
       
        end
        
    end
        
catch
   
	% Catch and display the last error or an exception thrown above
	err = lasterror;
	disp(err.message);
   
end