function pair_out = crossover(pair_in, pc)
% -------------------------------------------------
% CLASSICAL GENETIC ALGORITHM
% crossover - one-point crossover operator
%
% Made by:
% Daniel L. Kovacs
% <dkovacs@mit.bme.hu>
%
% Department of Measurement and Information Systems
% Faculty of Electrical Engineering and Informatics
% Budapest University of Technology and Economics
% 
% January 2010
% -------------------------------------------------

try
    
    % Initialize output variables
    pair_out = pair_in;

	% -----------------------------------------------------------------------------------------------------------------------------
    % -------------------------------------------------------- INPUT CHECK --------------------------------------------------------
    % -----------------------------------------------------------------------------------------------------------------------------
    
    % Check input variable 1/2
	if  ~(size(pair_in, 1) == 2) ||...                                 % IF the pair (pair_in) does not consist of exactly 2 individuals, OR...
        ~(size(pair_in, 2) >= 1),                                      % the individuals are of zero length, THEN...
       
        throw(MException('MATLAB:crossover:pair_in',...
                         'Input error (1/2): the pair of individuals should be a 2*N matrix (N>=1)'));
        
	end
    
	% Check input variable 2/2
    if ~all(eq(size(pc), [1, 1])) ||...                                 % IF the crossover probability (pc) is not a scalar, OR...
       ~(pc >= 0) ||...                                                 % it is negative, OR...
       ~(pc <= 1),                                                      % it is greater, than 1, THEN...
       
        throw(MException('MATLAB:crossover:pc',...
                         'Input error (2/2): the crossover probability should be a double between 0 and 1'));
        
    end
    
    % --------------------------------------------------------------------------------------------------------------------------------
    % -------------------------------------------------------- FUNCTION BODY  --------------------------------------------------------
    % --------------------------------------------------------------------------------------------------------------------------------
    
    % Decide whether to effectively cross the pair or leave it untouched
	if (rand <= pc)

        % Select the point from where the chromosomes exchange their genes
        cp          = 1 + round((size(pair_in, 2) - 1) * rand);

        % Cross the two chromosomes starting with their cp-th gene (1-point crossover)
        pair_out    = [[pair_in(1,1:(cp-1)), pair_in(2,cp:end)]; [pair_in(2,1:(cp-1)), pair_in(1,cp:end)]];

	end
        
catch
   
	% Catch and display the last error or an exception thrown above
	err = lasterror;
	disp(err.message);
   
end