function disp_timetable(t, params)
% -------------------------------------------------------
% CLASSICAL GENETIC ALGORITHM
% disp_timetable - display a timetable
%
% Made by:
% Daniel L. Kovacs
% <dkovacs@mit.bme.hu>
%
% Department of Measurement and Information Systems
% Faculty of Electrical Engineering and Informatics
% Budapest University of Technology and Economics
% 
% March 2010
% -------------------------------------------------------

try
    
	% -----------------------------------------------------------------------------------------------------------------------------
    % -------------------------------------------------------- INPUT CHECK --------------------------------------------------------
    % -----------------------------------------------------------------------------------------------------------------------------
    
    % Check input variable 1/2
	if  isempty(t),                         % IF the timetable (t) is empty, THEN...
       
        throw(MException('MATLAB:disp_timetable:t',...
                         'Input error (1/2): the timetable should not be empty'));
        
	end
    
    % Check input variable 2/2
	if  isempty(params),                  % IF the parameters of the timetable (params) are empty, THEN...                                                      
       
        throw(MException('MATLAB:disp_timetable:params',...
                         'Input error (2/2): the parameters of the timetable should not be empty'));
        
	end
    
    % --------------------------------------------------------------------------------------------------------------------------------
    % -------------------------------------------------------- FUNCTION BODY  --------------------------------------------------------
    % --------------------------------------------------------------------------------------------------------------------------------
       
    % The number of possible classrooms
    M = size(params.classroom, 1);
    
    % Decode all the starttimes and classrooms associated with the lectures in t
    starttimes  = idivide(int32(t-1), M) + 1;
    classrooms  = mod(t-1, M) + 1;
    
    % Extract columns from params.starttime matrix (of size N*2)
    startday    = params.starttime(:, 1);
    starthour   = params.starttime(:, 2);
    
    % Convert the starttimes vector (of size 1*chr_length) to corresponding
    % hour and day vectors, which tell that on which day and in which hour
    % do the lectures begin
    startdays   = startday(starttimes);
    starthours	= starthour(starttimes);
    
    % Calculate the exact endhours of the lectures
    endhours    = starthours + params.lecture(:, 2);

    % Sort the lectures according to their starting time, and then their classroom
    [st, sti]   = sort(t);
    
    % Go through all the sorted lectures
    for j = 1:size(st, 2),

        % Identify the lecture ID
        i           = sti(j);
        
        % Get all the parameters of the event
        lday        = params.day{startdays(i), 1};                                      % Day of the event
        lstart      = [int2str(starthours(i)), ':15'];                                  % Starting time of the event
        lend        = [int2str(endhours(i)), ':00'];                                    % End time of the event
        lplace      = params.classroom{classrooms(i),1};                                % Place of the event
        lclass      = params.class{params.course{params.lecture(i, 1), 5}, 1};          % Name of the class participating at the event
        lname       = params.subject{params.course{params.lecture(i, 1), 1}, 1};        % Subject-name of the event
        lscode      = params.subject{params.course{params.lecture(i, 1), 1}, 2};        % Subject-code of the event
        lccode      = params.course{params.lecture(i, 1), 2};                           % Course-code of the event
        ltype       = params.coursetype{params.course{params.lecture(i, 1), 3}, 1};     % Type of the event
        lteachers   = params.course{params.lecture(i, 1), 4};                           % TeacherID's of the event

        % Print everything about the event (Day, Starttime, Endtime, Place, Lecture name, Subject-code, Lecture-type, Course-code, Class name, Lecturers' name)
        fprintf('%s, %s-%s, %s\n', lday, lstart, lend, lplace)
        fprintf('%s (%s) %s [%s kurzus]\n', lname, lscode, ltype, lccode)
        fprintf('%s\n', lclass)
        
        % Some initialization for printing the names of the teachers of the given lecture
        ltnum   = size(lteachers,2);
        
        % Print the names of the teachers of the given lecture
        for k = 1:ltnum,
           
            % Get the actual name of the teacher 
            tname   = params.teacher{lteachers(k), 1};
            
            % If this is the last element of the teacher-list
            if k < ltnum,
               
                fprintf('%s, ', tname)
                
            else
               
               fprintf('%s\n\n', tname) 
                
            end % END IF

        end % END FOR-K
        
    end % END FOR-J
        
catch
   
	% Catch and display the last error or an exception thrown above
	err = lasterror;
	disp(err.message);
   
end