function fval = f_diophantos(x, params)
% -------------------------------------------------
% CLASSICAL GENETIC ALGORITHM
% f_diophantos - calculate the fitness value of a diophantos' equation
%                (the higher the fitness of such an equation the
%                 less it's value is in homogenous form)
%
% Made by:
% Daniel L. Kovacs
% <dkovacs@mit.bme.hu>
%
% Department of Measurement and Information Systems
% Faculty of Electrical Engineering and Informatics
% Budapest University of Technology and Economics
% 
% January 2010
% -------------------------------------------------

try
    
    % Initialize variables
    k       = size(x, 1);
    fval	= zeros(k, 1);

	% -----------------------------------------------------------------------------------------------------------------------------
    % -------------------------------------------------------- INPUT CHECK --------------------------------------------------------
    % -----------------------------------------------------------------------------------------------------------------------------
    
    % Check input variable 1/2
	if  isempty(x),                                         	% IF the matrix of individuals (x) is empty, THEN...
       
        throw(MException('MATLAB:f_diophantos:x',...
                         'Input error (1/2): the matrix of individuals should not be empty'));
        
	end
    
    % Check input variable 2/2
	if  isempty(params) ||...                                   % IF the input parameters struct is empty, OR...
        ~isfield(params, 'coeffs') ||...                        % it hasn't got a "coeffs" field, OR...
        ~isfield(params, 'powers') ||...                        % it hasn't got a "powers" field, OR...
        ~isfield(params, 'c') ||...                             % it hasn't got a "c" field, OR...
        prod(size(params.c)) ~= 1 ||...                         % the "c" field isn't a scalar, OR...
        ~all(eq(size(params.coeffs), [1, size(x, 2)])) ||...    % the "coeffs" field isn't a row-vector with as many columns as x has, OR...
        ~all(eq(size(params.coeffs), size(params.powers))),  	% the "powers" field isn't of the same size as the "coeffs" field, THEN...                                                    
       
        throw(MException('MATLAB:f_diophantos:params',...
                         'Input error (2/2): fitness parameters should have a "coeffs" and "powers" field, both row vectors, with the length of chromosomes'));
        
	end
    
    % --------------------------------------------------------------------------------------------------------------------------------
    % -------------------------------------------------------- FUNCTION BODY  --------------------------------------------------------
    % --------------------------------------------------------------------------------------------------------------------------------

    % Calculate the value of every diophantos' equation
    fval = (1 + abs(params.c + (x.^repmat(params.powers, k, 1))*params.coeffs')).^-1; % equals 1/(1+d), where d=abs(params.c + (x.^repmat(params.powers, k, 1))*params.coeffs')
  
catch
   
	% Catch and display the last error or an exception thrown above
	err = lasterror;
	disp(err.message);
   
end