function fval = f_neurals(x, params)
% -------------------------------------------------
% CLASSICAL GENETIC ALGORITHM
% f_neurals - calculate the fitness value of chromosomes
%             corresponding to neural network structures
%
% Made by:
% Daniel L. Kovacs
% <dkovacs@mit.bme.hu>
%
% Department of Measurement and Information Systems
% Faculty of Electrical Engineering and Informatics
% Budapest University of Technology and Economics
% 
% January 2010
% -------------------------------------------------

try
    
    % Initialize variables
    k       = size(x, 1);
    fval	= zeros(k, 1);

	% -----------------------------------------------------------------------------------------------------------------------------
    % -------------------------------------------------------- INPUT CHECK --------------------------------------------------------
    % -----------------------------------------------------------------------------------------------------------------------------
    
    % Check input variable 1/2
	if  isempty(x),                                         	% IF the matrix of individuals (x) is empty, THEN...
       
        throw(MException('MATLAB:f_neurals:x',...
                         'Input error (1/2): the matrix of individuals should not be empty'));
        
	end
    
    % Check input variable 2/2
	if  isempty(params) ||...                               	% IF the input parameters struct is empty, OR...
        ~isfield(params, 'p') ||...                           	% it hasn't got a "p" field (neural inputs), OR...
        ~isfield(params, 't') ||...                             % it hasn't got a "t" field (neural targets), OR...
        isempty(params.p) ||...                               	% the "p" field is empty, OR...
        isempty(params.t) ||...                               	% the "t" field is empty, OR...
        size(params.p, 2) ~= size(params.t, 2),               	% the size of "p" and "t" (the number of learning points) is not appropriate, THEN...                                                      
       
        throw(MException('MATLAB:f_neurals:params',...
                         'Input error (2/2): neural parameters must have a non-empty "p" and a "t" field (inputs and outputs) with the same number of columns'));
        
	end
    
    % --------------------------------------------------------------------------------------------------------------------------------
    % -------------------------------------------------------- FUNCTION BODY  --------------------------------------------------------
    % --------------------------------------------------------------------------------------------------------------------------------

    % Calculate the fitness of every individual
	for i=1:k,
        
        % Take the i-th individual from x
        ind = x(i, :);
        
        % Leave only the positive gene values from that individual
        ind = ind(find(ind > 0));	%#ok<FNDSB>
        
        if ~isempty(ind),           % IF the "remains" of the i-th individual (after pruning) are not empty
            
            % Build the PHENOTYPE (neural network) from the pre-processed GENOTYPE (chromosome)
            net                     = newff(params.p, params.t, ind);
            
            % Divide the training, test, and validation points systematically (for better comparison)
            net.divideFcn           = 'divideint';
            
            % Increase the number of epochs (from default 5) where the non-decreasing manner of the MSE (Mean Square Error) on validation points is tolerated during learning
            net.trainParam.max_fail = 10;
            
            % Turn of visualization of the progress of the learning process
            net.trainParam.show     = NaN;
            
            % Train the network (the phenotype)
            net                     = train(net, params.p, params.t);
            
            % Return the normalized MSE of the trained network as fitness (if MSE = 0, then FVAL = 1; if MSE = inf, then FVAL = 0)
            fval(i, 1)              = 1 / (1 + mse(params.t - sim(net, params.p)) + 0.02*sum(ind));
            
        end
        
	end
  
catch
   
	% Catch and display the last error or an exception thrown above
	err = lasterror;
	disp(err.message);
   
end