function fval = f_neuralw2(x, params)
% -------------------------------------------------
% CLASSICAL GENETIC ALGORITHM
% f_neuralw2 - calculate the fitness value of chromosomes
%              corresponding to neural network weights/biases
%              [optimized version]
%
% Made by:
% Daniel L. Kovacs
% <dkovacs@mit.bme.hu>
%
% Department of Measurement and Information Systems
% Faculty of Electrical Engineering and Informatics
% Budapest University of Technology and Economics
% 
% January 2010
% -------------------------------------------------

try
    
    % Initialize variables
    k       = size(x, 1);
    fval	= zeros(k, 1);

	% -----------------------------------------------------------------------------------------------------------------------------
    % -------------------------------------------------------- INPUT CHECK --------------------------------------------------------
    % -----------------------------------------------------------------------------------------------------------------------------
    
    % Check input variable 1/2
	if  isempty(x),                                         	% IF the matrix of individuals (x) is empty, THEN...
       
        throw(MException('MATLAB:f_neuralw2:x',...
                         'Input error (1/2): the matrix of individuals should not be empty'));
        
	end
    
    % Check input variable 2/2
	if  isempty(params) ||...                               	% IF the input parameters struct is empty, OR...
        ~isfield(params, 'p') ||...                           	% it hasn't got a "p" field (neural inputs), OR...
        ~isfield(params, 't') ||...                             % it hasn't got a "t" field (neural targets), OR...
        ~isfield(params, 'layers') ||...                     	% it hasn't got a "layers" field (hidden layers' neuron number), OR...
        isempty(params.p) ||...                               	% the "p" field is empty, OR...
        isempty(params.t) ||...                               	% the "t" field is empty, OR...
        isempty(params.layers) ||...                            % the "layers" field is empty, OR...
        size(params.p, 2) ~= size(params.t, 2),               	% the size of "p" and "t" (the number of learning points) is not appropriate, THEN...                                                      
       
        throw(MException('MATLAB:f_neuralw2:params',...
                         'Input error (2/2): neural parameters must have a non-empty "p", "t", "layers", and "net" field (inputs and outputs with the same number of columns; hidden network layers'' neuron number; neural network structure)'));
        
	end
    
    % --------------------------------------------------------------------------------------------------------------------------------
    % -------------------------------------------------------- FUNCTION BODY  --------------------------------------------------------
    % --------------------------------------------------------------------------------------------------------------------------------

    % Calculate the input/output dimensions
    sp      = size(params.p, 1);
    st      = size(params.t, 1);
    sp2     = size(params.p, 2);
    layers  = params.layers;
    z      	= ones(1, sp2);
    oc      = mat2cell(ones(k,sp2), ones(1,k), sp2);

	% The number of hidden layers (not including the output layer)
    l                   = size(layers, 2);
    
    b                   = (1 + sp) * layers(1);   
    m                   = reshape(x(:,1:b)', layers(1), k*(1+sp));                                          % layers(1) * (k*(1+sp)) matrix
    mc                  = mat2cell(m, layers(1), (1+sp)*ones(1, k));                                        % Divide m into k layers(1)*(1+sp) size cells (in a row)
    y                   = tansig(cell2mat(mc') * [params.p;z]);                                             % Output of the 1st layers: (k*layers(1)) * sp2 size matrix

    % Lets get through the hidden layers of every network in the population
	for i=2:l,
        
        a               = b;
        b               = b + (1 + layers(i-1)) * layers(i);       
      	m               = reshape(x(:,a+1:b)', layers(i), k*(1 + layers(i-1)));                          	% layers(i) * (k*(1+layers(i-1))) matrix
        mc              = mat2cell(m, layers(i), (1 + layers(i-1))*ones(1, k));                             % Divide m into k layers(i)*(1+layers(i-1)) size cells (in a row)
        yc              = mat2cell(y, layers(i-1)*ones(1, k), sp2);                                         % Divide y into k layers(i-1)*sp2 size cells
        y           	= tansig(blkdiag(mc{:}) * cell2mat(cellfun(@(a,b) [a;b], yc, oc, 'uni', false)));	% Output of the i-th layers: (k*layers(i)) * sp2 size matrix

	end
    
    a                   = b;
    b                   = b + (1 + layers(l)) * st;
 	m                   = reshape(x(:,a+1:b)', st, k*(1 + layers(l)));                                      % st * (k*(1+layers(l))) matrix   
    mc                  = mat2cell(m, st, (1 + layers(l))*ones(1, k));                                      % Divide m into k st*(1+layers(l)) size cells (in a row)
    yc                  = mat2cell(y, layers(l)*ones(1, k), sp2);                                           % Divide y into k layers(l)*sp2 size cells
    y                   = blkdiag(mc{:}) * cell2mat(cellfun(@(a,b) [a;b], yc, oc, 'uni', false));           % Output of the (l+1)-th layers: (k*st) * sp2 size matrix

    % Produce all the fitness values - in a k*1 column vector - for all the networks in the population according to their MSE (Mean Square Error)
    fval = cell2mat(cellfun(@(a) 1/(1+a), cellfun(@mse, mat2cell(kron(ones(k,1), params.t)-y, st*ones(1, k), sp2), 'uni', false), 'uni', false));

catch
   
	% Catch and display the last error or an exception thrown above
	err = lasterror;
	disp(err.message);
   
end