function fval = f_timetable(x, params)
% -------------------------------------------------------
% CLASSICAL GENETIC ALGORITHM
% f_timetable - calculate the fitness value of timetables
%
% Made by:
% Daniel L. Kovacs
% <dkovacs@mit.bme.hu>
%
% Department of Measurement and Information Systems
% Faculty of Electrical Engineering and Informatics
% Budapest University of Technology and Economics
% 
% March 2010
% -------------------------------------------------------

try
    
    % Initialize variables
    [k, n]  = size(x);
    fval	= zeros(k, 1);

	% -----------------------------------------------------------------------------------------------------------------------------
    % -------------------------------------------------------- INPUT CHECK --------------------------------------------------------
    % -----------------------------------------------------------------------------------------------------------------------------
    
    % Check input variable 1/2
	if  isempty(x),                         % IF the matrix of individuals (x) is empty, THEN...
       
        throw(MException('MATLAB:f_timetable:x',...
                         'Input error (1/2): the matrix of individuals should not be empty'));
        
	end
    
    % Check input variable 2/2
	if  isempty(params),                  % IF the parameters of the fitness function (params) are empty, THEN...                                                      
       
        throw(MException('MATLAB:f_timetable:params',...
                         'Input error (2/2): the parameters of the fitness function should not be empty'));
        
	end
    
    % --------------------------------------------------------------------------------------------------------------------------------
    % -------------------------------------------------------- FUNCTION BODY  --------------------------------------------------------
    % --------------------------------------------------------------------------------------------------------------------------------
       
    % The number of possible class-rooms
    M = size(params.classroom, 1);
    
    % Decode all the start-times and class-rooms associated with the lectures in the chromosomes of the given generation X
    starttimes  = idivide(int32(x-1), M) + 1;
    classrooms  = mod(x-1, M) + 1;
    
    % Extract some columns from params.starttime matrix (of size N*2)
    startday    = params.starttime(:, 1);
    starthour   = params.starttime(:, 2);
    
    % Convert the starttimes matrix (of size k*chr_length) to corresponding hour and day matrices, which tell that on which day and in which hour do the lectures of all the chromosomes in the population begin
    startdays   = startday(starttimes);
    starthours	= starthour(starttimes);
    
    % Calculate the endhours of the lectures
    endhours    = starthours + (ones(k, 1) * (params.lecture(:, 2) - 1)');

    % Prepare the data for the calculation of errors (collisions in timetables)
    nn          = n * ones(1, n);
    eh          = repcols(endhours, nn);
    
    % FITNESS FACTOR #1 [PRIMARY]: Calculate the number of collisions ONCE between lectures in time and space (for every individual chromosome in the population)...
    fp1         = sum((eh - repmat(starthours, 1, n) >= 0) .* ...                           % IF the ending of a lecture is later than an other's beginning, AND...
                       (eh - repmat(endhours, 1, n) <= 0) .* ...                            % IF the ending of a lecture is earlier than an other's ending, AND... 
                       (repcols(startdays, nn) - repmat(startdays, 1, n) == 0) .* ...       % IF the two lectures start on the same day, AND... 
                       (repcols(classrooms, nn) - repmat(classrooms, 1, n) == 0), 2) - ...  % IF the two lectures are in the same classroom.
                  n * ones(k, 1);                                                           % Subtract this because of the self-collisions

    % Some manual memory management
    clear nn eh
              
    % Extract the capacity of every possible classroom (into a column-vector)
    ccapacity   = cell2mat(params.classroom(:, 3));
    
    % Convert the classrooms matrix (of size k*chr_length) to corresponding classroom-capacities matrix
    ccapacities = ccapacity(classrooms);
    
    % Number of students on the different lectures
    studentsnum = ones(k, 1) * params.lecture(:, 3)';
    
    % FITNESS FACTOR #2 [PRIMARY]: Calculate the number of lectures where students don't fit into the associated classroom (for every individual chromosome in the population)...
    fp2          = sum((ccapacities - studentsnum) < 0, 2);
    
    % The weights of primary fitness factors (number of colliding and overly dense lectures, etc) 
    fpw         = [0.8; 0.2];

    % Calculate the weighted sum of primary fitness factors (to minimize them together) - "fpw" is the column vector of corresponding weights 
    fp          = [fp1, fp2] * fpw;
    
    % Calculate the PRIMARY FITNESS value [0..1] of every individual timetable in the population 
    f           = 1 ./ (1 + fp);

    % An index for overall fitness calculations (indicating those individuals whose primary fitness is optimal) 
    fsi         = (f == 1);
    
    % FITNESS FACTOR #3 [SECONDARY]: Calculate the effectiveness of classroom usage (for every individual chromosome in the population)...
    fs1          = sum(studentsnum ./ ccapacities, 2);

    % The weights of secondary fitness factors (effectiveness of classroom usage, etc) 
    fsw         = [1/n];

    % Calculate the weighted sum of secondary fitness factors - "fsw" is the column vector of corresponding weights 
    fs          = [fs1] * fsw;
    
    % Increase the primary fitness with SECONDARY FITNESS of those individuals whose primary fitness is optimal 
    f           = f + (fsi .* fs);

    % Return the overall fitness of each individual in the population
    fval        = f;
        
catch
   
	% Catch and display the last error or an exception thrown above
	err = lasterror;
	disp(err.message);
   
end