function [best, history] = ga(pop_size, chr_length, genes, pc, pm, fitness, stopcrit)
% -------------------------------------------------
% CLASSICAL GENETIC ALGORITHM
% ga - genetic algorithm core
%
% Made by:
% Daniel L. Kovacs
% <dkovacs@mit.bme.hu>
%
% Department of Measurement and Information Systems
% Faculty of Electrical Engineering and Informatics
% Budapest University of Technology and Economics
% 
% January 2010
% -------------------------------------------------

try
    
    % Initialize output variables
    best            = '';
    history.gen     = 0;    % Zero generations were "born" until now
    history.bestf 	= [];   % Best fitness over generations
    history.best  	= [];   % Best individuals over generations
    history.avgf 	= [];   % Average fitness over generations
    
    % Notice: the length of the history depends on the stop criteria and
    % other stochastic parameters and thus its length cannot be determined
    % exactly now, i.e. it cannot be pre-allocated for faster performance...
    % Nonetheless with a careful analysis of the stop criteria an
    % appropriate overestimate could be given (and so a sub-optimal
    % pre-allocation would become possible). Performance issues off.
    
    % -----------------------------------------------------------------------------------------------------------------------------
    % -------------------------------------------------------- INPUT CHECK --------------------------------------------------------
    % -----------------------------------------------------------------------------------------------------------------------------
    
    % Check input variable 1/7
	if ~all(eq(size(pop_size), [1, 1])) ||...                           % IF the size of the population (pop_size) is not a scalar, OR...
       ~(pop_size >= 2)  ||...                                          % pop_size is less, than 2, OR...
       ~(mod(pop_size, 2) == 0),                                        % pop_size is not even, THEN...
       
        throw(MException('MATLAB:ga:pop_size',...
                         'Input error (1/7): the size of the population should be a positive, even integer'));
        
	end
    
    % Check input variable 2/7
    if ~all(eq(size(chr_length), [1, 1])) ||...                         % IF the length of the chromosomes (chr_length) is not a scalar, OR...
       ~(chr_length >= 1),                                              % chr_length is less, than 2, THEN...
       
        throw(MException('MATLAB:ga:chr_length',...
                         'Input error (2/7): the length of chromosomes should be a positive integer'));
        
    end
    
    % Check input variable 3/7
    if isempty(genes) ||...                                             % IF the gene-definition (genes) is empty, OR...
       ~isfield(genes, 'domain') ||...                                  % it hasn't got a "domain" field, OR...
       ~isfield(genes, 'discrete') ||...                                % it hasn't got a "discrete" field, OR...
       isempty(genes.domain) ||...                                      % it's "domain" field is empty, OR...
       isempty(genes.discrete) ||...                                    % it's "discrete" field is empty, OR...
       ~all(eq(size(genes.domain), [1, 2])) ||...                   	% the "domain" field isn't an 1*2 row vector, OR...
       ~(genes.domain(2) > genes.domain(1)),                           	% it's 2nd element is not greater, than the 1st, THEN...
       
        throw(MException('MATLAB:ga:genes',...
                         'Input error (3/7): the definition of genes should be given as a struct with a non-empty "domain" and "discrete" field (the "discrete" field is boolean, while the "domain" field should be a row-vector of two doubles [A,B], where B > A)'));
        
    end
    
	% Check input variable 4/7
    if ~all(eq(size(pc), [1, 1])) ||...                                 % IF the crossover probability (pc) is not a scalar, OR...
       ~(pc >= 0) ||...                                                 % it is negative, OR...
       ~(pc <= 1),                                                      % it is greater, than 1, THEN...
       
        throw(MException('MATLAB:ga:pc',...
                         'Input error (4/7): the crossover probability of two individuals should be a double between 0 and 1'));
        
    end
    
	% Check input variable 5/7
    if ~all(eq(size(pm), [1, 1])) ||...                                 % IF the mutation probability (pm) is not a scalar, OR...
       ~(pm >= 0) ||...                                                 % it is negative, OR...
       ~(pm <= 1),                                                      % it is greater, than 1, THEN...
       
        throw(MException('MATLAB:ga:pm',...
                         'Input error (5/7): the mutation probability of a gene should be a double between 0 and 1'));
        
    end
    
	% Check input variable 6/7
    if ~isstruct(fitness) ||...                                         % IF the fitness definition (fitness) isn't a structure, OR...
       ~isfield(fitness, 'fname') ||...                                 % it hasn't got an "fname" field, OR...
       ~isfield(fitness, 'fparams') ||...                               % it hasn't got an "fparams" field, OR...
       isempty(fitness.fname) ||...                                     % the "fname" field is empty, OR...
       ~ischar(fitness.fname) ||...                                     % the "fname" field isn't a string, OR...
       ~exist(fitness.fname, 'file'),                                   % the function M-file referenced by the "fname" field doesn't exist, THEN...
       
        throw(MException('MATLAB:ga:fitness',...
                         'Input error (6/7): the fitness should be a structure with a non-empty "fname" string-field (referencing an existing function, i.e. M-file) and an arbitrary "fparams" field'));
        
    end
    
	% Check input variable 7/7
    scerr = false;
    
    if  ~(size(stopcrit, 1) == 1) ||...                                 % IF stop criteria (stopcrit) aren't given in a row vector, OR...
        ~(size(stopcrit, 2) >= 1),                                      % the row vector is empty, THEN...
    
        scerr = true;
    
    else

        % There should be maximally 1 stop-criterion of every type
        scnum.mingennum	= 0;
        scnum.minbestf	= 0;
        scnum.minavgf	= 0;
        
        for i=1:size(stopcrit, 2),
        
            if ~isstruct(stopcrit(i)) ||...                             % IF the i-th stop criterion (stopcrit(i)) isn't a structure, OR...
               ~isfield(stopcrit(i), 'ctype') ||...                     % it hasn't got a "ctype" field, OR...
               (~strcmp(stopcrit(i).ctype, 'mingennum') &&...           % its "ctype" field...
                ~strcmp(stopcrit(i).ctype, 'minbestf') &&...            % has a value...
                ~strcmp(stopcrit(i).ctype, 'minavgf')) ||...            % different from 'mingennum' OR 'minbestf' OR 'minavgf', OR...
               scnum.(stopcrit(i).ctype) ~= 0 ||...                     % there was already a j-th (j < i) stop criterion with the same "ctype", OR...
               ~isfield(stopcrit(i), 'cparams') ||...                   % the i-th stop criterion hasn't got a "cparams" field, OR...
               isempty(stopcrit(i).cparams),                            % or that "cparams" field is empty, THEN...
           
                scerr = true;
                break;
           
            end
            
            % Increment the number of the criteria having the type of
            % the i-th stop criterion among the stop criteria
            scnum.(stopcrit(i).ctype) = scnum.(stopcrit(i).ctype) + 1;
            
        end
        
    end
    
    % IF there was some error concerning given stop criteria, THEN...
    if scerr,
        
        throw(MException('MATLAB:ga:stopcrit',...
                         'Input error (7/7): the stop-criteria should be given as an 1*N array (N >= 1) with structures as elements having distinct, non-empty "ctype" string-fields (with a value being either "mingennum", or "minbestf", or "minavgf") and arbitrary, but non-empty "cparams" fields'));
        
    end
    
    % --------------------------------------------------------------------------------------------------------------------------------
    % -------------------------------------------------------- FUNCTION BODY  --------------------------------------------------------
    % --------------------------------------------------------------------------------------------------------------------------------
    
    % GENERATION >>> Generate a random (uniformly distributed) initial population (KxN)
    pop     = initpop(pop_size, chr_length, genes);
    
    % FITNESS >>> Calculate the fitness value of all the individuals in the initial population (column vector)
    fvals   = feval(fitness.fname, pop, fitness.fparams);
    
    % Increment the number of generations created
    history.gen     = history.gen + 1;
    disp(['Generation #', num2str(history.gen)])
    
    % Get the current best fitness (and its index) of the initial population
    [bestf, besti]	= max(fvals);
    
    % Save the current best fitness value
    history.bestf 	= [history.bestf, bestf];
    disp(['Best fitness: ', num2str(history.bestf(history.gen), 16)])

    % Get and save the current best individual of the initial population
    best            = pop(besti, :);
    history.best  	= [history.best; best];
        
    % Get and save the current average fitness of the initial population
    history.avgf 	= [history.avgf, median(fvals, 1)];
    disp(['Avg. fitness: ', num2str(history.avgf(history.gen), 16)])
    
    % Initialize plot
    xlabel('Generations')
    ylabel('Fitness')
    title('Best and average fitness over generations')
    set(gcf, 'Name', 'Classical Genetic Algorithm')
    set(gcf, 'NumberTitle', 'off')
    hold on

    % STOP CRITERIA >>> WHILE all the stop criteria are all false...
    while ~check_stopcrit(history, stopcrit),
       
        % ELITISM >>> Implement elitism (by putting the best individual of the current generation into the next generation without any change
        newpop = best;
        
        % Select pop_size/2 "lucky" pairs and possibly cross and mutate them before putting them into the next generation
        for i=1:pop_size/2,
      
            % SELECTION >>> Select a pair of parents for the next generation
            pair = selection(pop, fvals);

            % CROSSOVER >>> Cross the pair (1-point crossover)
            pair = crossover(pair, pc);
            
            % MUTATION >>> Mutate the parents (every gene with pm probability)
            pair = mutation(pair, pm, genes);
        
            % Add the newly created pair of individuals to the next generation
            newpop = [newpop; pair]; %#ok<AGROW>
            
        end
        
        % Let the next generation become current
        pop = newpop;
        
     	% FITNESS >>> Calculate the fitness value of all the individuals in the current generation (column vector)
        fvals   = feval(fitness.fname, pop, fitness.fparams);

        % Increment the number of generations created
        history.gen     = history.gen + 1;
        clc, fprintf('\nGeneration #%d\n', history.gen)
        
        % Get the current best fitness (and its index) of the current generation
        [bestf, besti]	= max(fvals);
    
        % Save the current best fitness value
        history.bestf 	= [history.bestf, bestf];
        disp(['Best fitness: ', num2str(history.bestf(history.gen), 16)])
 
        % Get and save the current best individual of the current generation
        best            = pop(besti, :);
        history.best  	= [history.best; best];
        
        % Get and save the current average fitness of the current generation
        history.avgf 	= [history.avgf, median(fvals, 1)];
        disp(['Avg. fitness: ', num2str(history.avgf(history.gen), 16)])
            
    end
        
	% Update plot (best and average fitness over generations)
	plot(1:history.gen, history.bestf, 'b', 1:history.gen, history.avgf, ':r')
	%drawnow;
    
catch
   
	% Catch and display the last error or an exception thrown above
	err = lasterror;
	disp(err.message);
   
end