function pop = initpop(pop_size, chr_length, genes)
% -------------------------------------------------
% CLASSICAL GENETIC ALGORITHM
% initpop - random initialization of the population 
%
% Made by:
% Daniel L. Kovacs
% <dkovacs@mit.bme.hu>
%
% Department of Measurement and Information Systems
% Faculty of Electrical Engineering and Informatics
% Budapest University of Technology and Economics
% 
% January 2010
% -------------------------------------------------

try
    
    % Initialize output variables
    pop = '';

    % -----------------------------------------------------------------------------------------------------------------------------
    % -------------------------------------------------------- INPUT CHECK --------------------------------------------------------
    % -----------------------------------------------------------------------------------------------------------------------------
        
    % Check input variable 1/3
	if ~all(eq(size(pop_size), [1, 1])) ||...                   % IF the size of the population (pop_size) is not a scalar, OR...
       ~(pop_size >= 2)  ||...                                  % pop_size is less, than 2, OR...
       ~(mod(pop_size, 2) == 0),                                % pop_size is not even, THEN...
       
        throw(MException('MATLAB:initpop:pop_size',...
                         'Input error (1/3): the size of the population should be a positive, even integer'));
        
	end
    
    % Check input variable 2/3
    if ~all(eq(size(chr_length), [1, 1])) ||...                 % IF the length of the chromosomes (chr_length) is not a scalar, OR...
       ~(chr_length >= 1),                                      % chr_length is less, than 2, THEN...
       
        throw(MException('MATLAB:initpop:chr_length',...
                         'Input error (2/3): the length of chromosomes should be a positive integer'));
        
    end
    
    % Check input variable 3/3
    if isempty(genes) ||...                                             % IF the gene-definition (genes) is empty, OR...
       ~isfield(genes, 'domain') ||...                                  % it hasn't got a "domain" field, OR...
       ~isfield(genes, 'discrete') ||...                                % it hasn't got a "discrete" field, OR...
       isempty(genes.domain) ||...                                      % it's "domain" field is empty, OR...
       isempty(genes.discrete) ||...                                    % it's "discrete" field is empty, OR...
       ~all(eq(size(genes.domain), [1, 2])) ||...                   	% the "domain" field isn't an 1*2 row vector, OR...
       ~(genes.domain(2) > genes.domain(1)),                           	% it's 2nd element is not greater, than the 1st, THEN...
       
        throw(MException('MATLAB:initpop:genes',...
                         'Input error (3/3): the definition of genes should be given as a struct with a non-empty "domain" and "discrete" field (the "discrete" field is boolean, while the "domain" field should be a row-vector of two doubles [A,B], where B > A)'));
        
    end
    
    % --------------------------------------------------------------------------------------------------------------------------------
    % -------------------------------------------------------- FUNCTION BODY  --------------------------------------------------------
    % --------------------------------------------------------------------------------------------------------------------------------
    
    % Generate a random (uniformly distributed) initial population
    pop = genes.domain(1) + ((genes.domain(2) - genes.domain(1)) .* rand(pop_size, chr_length));
    
    % IF the gene values are discrete, round the previously generated values
    if genes.discrete,
        
        pop = round(pop);
        
    end
        
catch
   
	% Catch and display the last error or an exception thrown above
	err = lasterror;
	disp(err.message);
   
end