function [best, history] = main(probdom)
% -------------------------------------------------
% CLASSICAL GENETIC ALGORITHM
% main - test script for the genetic algorithm
%
% Made by:
% Daniel L. Kovacs
% <dkovacs@mit.bme.hu>
%
% Department of Measurement and Information Systems
% Faculty of Electrical Engineering and Informatics
% Budapest University of Technology and Economics
% 
% March 2010
% -------------------------------------------------


try

    % Clear the command window
    clc;
    
    % Print a welcome message
   	fprintf('==================================\n')
  	fprintf('Classical Genetic Algorithm (v1.0)\n')
 	fprintf('==================================\n')
  	fprintf('(c) Daniel L. Kovacs\n')
    fprintf('    dkovacs@mit.bme.hu\n')
    fprintf('    ISRG (Intelligent Systems Research Group)\n')
    fprintf('    Department of Measurement and Information Systems\n')
   	fprintf('    Budapest University of Technology and Economics\n')
  	fprintf('    2010 March\n\n')
    
    % Set this just for the sake of precise output
    format long;
    
    % Enable implicit multiprocessing (if possible)
    maxNumCompThreads('automatic');
    
    % Initialize random number generator
    % rand('twister', sum(100*clock));
    
    % Check whether the is only 1 input argument
    if nargin ~= 1,
    
        fprintf('The main script should be called with only one input parameter\n')
        fprintf('that specifies the problem domain where GA is used:\n')
        fprintf('\n')
        fprintf('\tmain(probdom);\n')
        fprintf('\n')
        fprintf('probdom = ''bin'' \t\t(for an evolution of binary chromosomes)\n')
        fprintf('probdom = ''dio'' \t\t(for an evolution of solutions to diophantos'' equations)\n')
        fprintf('probdom = ''netw''\t\t(for an evolution of neural network weights/biases)\n')
        fprintf('probdom = ''nets''\t\t(for an evolution of neural network structures)\n')
        fprintf('probdom = ''tmt'' \t\t(for an evolution of timetables)\n\n')

    else
    
        switch probdom

            case 'bin'

                % Welcome message
                fprintf('-------------------------------\nEvolution of binary chromosomes\n-------------------------------\n\n')

                % Set the size of the population
                pop_size       	= 30;

                % Set the length of the individuals (chromosomes, gene-chains)
                chr_length   	= 20;

                % Set the domain of gene values to discrete {0,1}
                genes.domain   	= [0, 1];
                genes.discrete  = true;

                % Set the probability of crossover
                pc            	= 0.6;

                % Set the probability of gene-mutation
                pm           	= 0.05;

                % Define the fitness
                fitness.fname  	= 'f_binary';               % Reference to a concrete fitness function F_BINARY.M
                fp.range        = [0, (2^chr_length) - 1];  % FP is a struct of its input parameters; FP.RANGE is one of them: the range of the fitness func.
                fitness.fparams	= fp;                       % FITNESS.FPARAMS is optional (inputs of the fitness function - can be anything, really)
                                                            % Note: FP.RANGE is an optional parameter-field for F_BINARY.
                                                            % If set, fitness values are normalized (between 0 and 1)
                % Define the stop criteria
                s1.ctype      	= 'mingennum';              % A limit on the number of generations
                s1.cparams    	= 200;
                s2.ctype      	= 'minbestf';               % A limit on the best fitness
                s2.cparams      = 1;                        % This should depend on the actual range of the fitness function (e.g. normalized or not?)
                stopcrit        = [s1, s2];

                % Run the algorithm with the above parameters
                [best, history] = ga(pop_size, chr_length, genes, pc, pm, fitness, stopcrit);

                fprintf('\nBest individual of the last generation:\n')
                disp(num2str(best))
            
            case 'dio'

                % Welcome message
                fprintf('------------------------------------------------\nEvolution of solutions to diophantos'' equations\n------------------------------------------------\n\n')

                % Set the size of the population
                pop_size       	= 30;

                % Set the domain of gene values to a discrete interval
                genes.domain   	= [1, 10];
                genes.discrete  = true;

                % Set the probability of crossover
                pc            	= 0.6;

                % Set the probability of gene-mutation
                pm           	= 0.3;

                % Define the fitness
                fitness.fname  	= 'f_diophantos';         	% Reference to a concrete fitness function F_DIOPHANTOS.M
                fp.coeffs       = [1, 1, -1];               % The coefficients of the homogenous equation
                fp.powers       = [2, 2, 2];                % The powers to which the different variables are raised
                fp.c            = 0;                        % A constant to be added to (the left side of) the equation
                fitness.fparams	= fp;                       
                
                % Set the length of the individuals (chromosomes, gene-chains)
                chr_length   	= size(fp.coeffs, 2);
                
                % Define the stop criteria
                s1.ctype      	= 'mingennum';              % A limit on the number of generations
                s1.cparams    	= 1000;
                s2.ctype      	= 'minbestf';               % A limit on the best fitness
                s2.cparams      = 1;                        % This should depend on the actual range of the fitness function (e.g. normalized or not?)
                stopcrit        = [s1, s2];

                % Run the algorithm with the above parameters
                [best, history] = ga(pop_size, chr_length, genes, pc, pm, fitness, stopcrit);

                fprintf('\nBest individual of the last generation:\n')
                disp(num2str(best))
                
            case 'netw'

                % Welcome message
                fprintf('------------------------------------------\nEvolution of neural network weights/biases\n------------------------------------------\n\n')

                % Set the size of the population
                pop_size       	= 30;

                % Set the domain of gene values to discrete {0,10}
                genes.domain  	= [-7, 7];
                genes.discrete  = false;

                % Set the probability of crossover
                pc            	= 0.6;

                % Set the probability of gene-mutation
                pm           	= 0.01;

                % Define the fitness
                fitness.fname  	= 'f_neuralw';                              % Reference to a concrete fitness function

                % Special fitness parameters (p - input; t - target; layers - hidden layers' neuron-number)
                fp.p            = [-1:0.01:1];                              %#ok<NBRAK>
                %fp.t            = sin(2*pi*fp.p);                           % a sine wave
                fp.t            = exp(-5*fp.p.^2); 
                fp.layers       = [3,3];                                    % The structure of the feed-forward neural network (only the hidden layers without the output layer (which is a consequence of the target-dimension))) 
                fp.net          = newff(fp.p, fp.t, fp.layers);             % Create the neural network structure for later weight/bias modification
                fitness.fparams	= fp;

                % Set the length of the individuals (chromosomes systematically encoding neural network layer weights and biases)
                chr_length   	= (1 + size(fp.p, 1)) * fp.layers(1);
                for i=2:size(fp.layers, 2), chr_length  = chr_length + (1 + fp.layers(i-1)) * fp.layers(i); end
                chr_length      = chr_length + (1 + fp.layers(size(fp.layers, 2))) * size(fp.t, 1);                
                
                % Define the stop criteria
                s1.ctype      	= 'mingennum';                              % A limit on the number of generations
                s1.cparams    	= 600;
                s2.ctype      	= 'minbestf';                               % A limit on the best fitness
                s2.cparams      = 1;                                        % This should depend on the actual range of the fitness function (e.g. normalized or not?)
                stopcrit        = [s1, s2];

                % Run the algorithm with the above parameters
                [best, history] = ga(pop_size, chr_length, genes, pc, pm, fitness, stopcrit);

                % Display the best chromosome of the last generation of the evolution 
                fprintf('\nBest individual of the last generation:\n')
                disp(num2str(best))

                % Notify the User about what is going to happen 
                fprintf('\nTesting the best individual...\n')

             	% Convert that best chromosome into a corresponding neural network 
                net             = decode_chr2netw(best, newff(fp.p, fp.t, fp.layers), fp.layers, size(fp.p, 1), size(fp.t, 1));
                
                % Create and set a new figure to plot the output of network
                figure
                title(sprintf('Output of the best individual of generation no. %d', history.gen))
                set(gcf, 'Name', 'Genetic-Neural Network Simulation')
                set(gcf, 'NumberTitle', 'off')

                % Plot the output of the network corresponding to the best chromosome of the last generation of the evolution
                plot(fp.p, fp.t, fp.p, simnet(net, fp.p), 'x')
            
            case 'nets'

                % Welcome message
                fprintf('--------------------------------------\nEvolution of neural network structures\n--------------------------------------\n\n')

                % Set the size of the population
                pop_size       	= 4;

                % Set the length of the individuals (chromosomes, gene-chains)
                chr_length   	= 2;

                % Set the domain of gene values to discrete {0,10}
                genes.domain  	= [0, 10];
                genes.discrete  = true;

                % Set the probability of crossover
                pc            	= 0.6;

                % Set the probability of gene-mutation
                pm           	= 0.05;

                % Define the fitness
                fitness.fname  	= 'f_neurals';                              % Reference to a concrete fitness function F_NEURALS.M

                % Special fitness parameters (p - input; t - target)
                fp.p            = [-1:0.01:1];                              %#ok<NBRAK>
                fp.t            = sin(2*pi*fp.p) + 0.1*randn(size(fp.p));	% a noisy sine wave
                fitness.fparams	= fp;                                       % see. below (OTHERWISE CASE)!

                % Define the stop criteria
                s1.ctype      	= 'mingennum';                              % A limit on the number of generations
                s1.cparams    	= 20;
                s2.ctype      	= 'minbestf';                               % A limit on the best fitness
                s2.cparams      = 1;                                        % This should depend on the actual range of the fitness function (e.g. normalized or not?)
                stopcrit        = [s1, s2];

                % Run the algorithm with the above parameters
                [best, history] = ga(pop_size, chr_length, genes, pc, pm, fitness, stopcrit);

                % Display the best chromosome of the last generation of the evolution 
                fprintf('\nBest individual of the last generation:\n')
                disp(num2str(best))

                % Notify the User about what is going to happen 
                fprintf('\nTesting the best individual...\n')

                % Convert that best chromosome into a corresponding neural network structure and set some of it's parameters 
                net                     = newff(fp.p, fp.t, best(find(best > 0)));              %#ok<FNDSB>

                % Divide the training, test, and validation points
                % systematically (like during the algorithm)
                net.divideFcn           = 'divideint';
                
                % Increase the number of epochs (from default 5) where the non-decreasing manner of the MSE (Mean Square Error) on validation points is tolerated during learning (like during the algorithm) 
                net.trainParam.max_fail = 10;

                % Train the network (the phenotype) corresponding to the best chromosome (genotype) evolved during the previous run of the GA 
                net                     = train(net, fp.p, fp.t);

                % Create and set a new figure to plot the output of network
                figure
                title(sprintf('Output of the best individual of generation no. %d', history.gen))
                set(gcf, 'Name', 'Genetic-Neural Network Simulation')
                set(gcf, 'NumberTitle', 'off')

                % Plot the output of the network corresponding to the best chromosome of the last generation of the evolution
                plot(fp.p, fp.t, fp.p, sim(net, fp.p), 'x')

            case 'tmt'

                % Welcome message
                fprintf('------------------------------------------------\nEvolution of timetables\n------------------------------------------------\n\n')

                % Set the size of the population
                pop_size       	= 30;

                % Set the probability of crossover
                pc            	= 0.6;

                % Set the probability of gene-mutation
                pm           	= 0.2;

                % Define the fitness
                fitness.fname           = 'f_timetable';         	% Reference to a concrete fitness function F_TIMETABLE.M
                [fp, genes, chr_length] = f_timetable_params(1);    % Create a problem
                fitness.fparams         = fp;
                                
                % Define the stop criteria
                s1.ctype      	= 'mingennum';                      % A limit on the number of generations
                s1.cparams    	= 200;
                s2.ctype      	= 'minbestf';                    	% A limit on the best fitness
                s2.cparams      = 1.7;                            	% This should depend on the actual range of the fitness function (e.g. normalized or not?)
                stopcrit        = [s1, s2];

                % Run the algorithm with the above parameters
                [best, history] = ga(pop_size, chr_length, genes, pc, pm, fitness, stopcrit);

                fprintf('\nBest individual of the last generation:\n')
                disp(num2str(best))
                
                fprintf('\n-----------------------------------------------\nTimetable corresponding to the best individual:\n-----------------------------------------------\n\n')       
                disp_timetable(best, fp)
                
            otherwise
                    
                disp('Unknown switch!')

        end
        
    end

catch
   
	% Catch and display the last error or an exception thrown above
	err = lasterror;
	disp(err.message);
   
end