function y = mutation(x, pm, genes)
% -------------------------------------------------
% CLASSICAL GENETIC ALGORITHM
% mutation - mutation operator
%
% Made by:
% Daniel L. Kovacs
% <dkovacs@mit.bme.hu>
%
% Department of Measurement and Information Systems
% Faculty of Electrical Engineering and Informatics
% Budapest University of Technology and Economics
% 
% January 2010
% -------------------------------------------------

try
    
    % Initialize output variables
    y = x;

	% -----------------------------------------------------------------------------------------------------------------------------
    % -------------------------------------------------------- INPUT CHECK --------------------------------------------------------
    % -----------------------------------------------------------------------------------------------------------------------------
    
    % Check input variable 1/3
	if  ~(size(x, 1) >= 1) ||...                                 % IF there is not any input individual (x), OR...
        ~(size(x, 2) >= 1),                                      % the individuals are of zero length, THEN...
       
        throw(MException('MATLAB:mutation:x',...
                         'Input error (1/3): there should be at least 1 non-zero long individual to mutate'));
        
	end
    
	% Check input variable 2/3
    if ~all(eq(size(pm), [1, 1])) ||...                                 % IF the mutation probability (pm) is not a scalar, OR...
       ~(pm >= 0) ||...                                                 % it is negative, OR...
       ~(pm <= 1),                                                      % it is greater, than 1, THEN...
       
        throw(MException('MATLAB:mutation:pm',...
                         'Input error (2/3): the mutation probability should be a double between 0 and 1'));
        
    end
    
    % Check input variable 3/3
    if isempty(genes) ||...                                             % IF the gene-definition (genes) is empty, OR...
       ~isfield(genes, 'domain') ||...                                  % it hasn't got a "domain" field, OR...
       ~isfield(genes, 'discrete') ||...                                % it hasn't got a "discrete" field, OR...
       isempty(genes.domain) ||...                                      % it's "domain" field is empty, OR...
       isempty(genes.discrete) ||...                                    % it's "discrete" field is empty, OR...
       ~all(eq(size(genes.domain), [1, 2])) ||...                   	% the "domain" field isn't an 1*2 row vector, OR...
       ~(genes.domain(2) > genes.domain(1)),                           	% it's 2nd element is not greater, than the 1st, THEN...
       
        throw(MException('MATLAB:mutation:genes',...
                         'Input error (3/3): the definition of genes should be given as a struct with a non-empty "domain" and "discrete" field (the "discrete" field is boolean, while the "domain" field should be a row-vector of two doubles [A,B], where B > A)'));
        
    end
    
    % --------------------------------------------------------------------------------------------------------------------------------
    % -------------------------------------------------------- FUNCTION BODY  --------------------------------------------------------
    % --------------------------------------------------------------------------------------------------------------------------------
    
    % Get the number and length of the individuals to mutate
    [k, n] 	= size(x);
    
    % Create a boolean mutation-mask matrix for them
    mm      = pm * ones(k, n) >= rand(k, n);
    
    % Create a matrix of mutated values
    mv      = genes.domain(1) + ((genes.domain(2) - genes.domain(1)) .* rand(k, n));
       
    % IF the gene values are discrete, round the previously generated values
    if genes.discrete,
     
        mv = round(mv);
        
    else
        
        % Reduce the deviance in the continuous case...
        e   = 0.1;
        mv  = (1-e)*x + e*mv;
        
    end
    
    % Merge the mutated genes with the original genes left untouched to produce the output (the appropriately mutated individuals)
    y       = (mm .* mv) + (~mm .* x);
    
    % - Notice that this way a gene value from x may be "changed" to the same value again...
    % - A fast alternative would be to consider mutation to work with the following:
    %
    %     mv = (genes.domain(2) * ones(k, n)) - (x - (genes.domain(1) * ones(k, n)));
    %
    %   This would be a kind of "inversion" of mutated genes working only if genes.domain(1)>=0
    %   This condition holds now, but if it wouldn't, and genes.domain(1) < 0 would hold, then
    %   a simple shifting could solve the problem (adding abs(genes.domain(1)) to genes.domain and x),
    %   and then the above "inversion" could be used...
    % - The point about mutation in a genetic algorithm is that together
    %   with the crossover operator it should enable "reaching" any possible
    %   individual from any other with a positive probability (uniform
    %   distribution is preferred again). This is a key component of the
    %   convergence of the GA. If this "reachability" doesn't hold, then the
    %   algorithm is not necessarily convergent in general.

catch
   
	% Catch and display the last error or an exception thrown above
	err = lasterror;
	disp(err.message);
   
end