//******************************************************************************
//* Driver for the SD memory card.                                             *
//* Written by   : Tamas Raikovich                                             *
//* Version      : 1.0                                                         *
//* Last modified: 2012.10.11.                                                 *
//******************************************************************************
#include <xparameters.h>
#include <string.h>
#include "memory_card.h"
#include "..\spi\spi.h"
#include "..\timer\timer.h"


//******************************************************************************
//* MMC/SD card commands.                                                      *
//******************************************************************************
#define CMD0    (0x40 + 0)      //Memory card reset.
#define CMD1    (0x40 + 1)      //Memory card initialization (MMC).
#define CMD8    (0x40 + 8)      //Sends the interface condition.
#define CMD12   (0x40 + 12)     //Multiple block read transfer stop.
#define CMD16   (0x40 + 16)     //Sets the block length.
#define CMD17   (0x40 + 17)     //Single block read.
#define CMD18   (0x40 + 18)     //Multiple blocks read.
#define CMD24   (0x40 + 24)     //Single block write.
#define CMD25   (0x40 + 25)     //Multiple blocks write.
#define CMD55   (0x40 + 55)     //Next command is an application specific command.
#define CMD58   (0x40 + 58)     //Reads the OCR register.
#define CMD59   (0x40 + 59)     //Enables/disables the CRC check.
#define	ACMD23  (0xC0 + 23)     //Sets the number of block to be pre-erased before write.
#define	ACMD41  (0xC0 + 41)     //Memory card initialization (SD).


//******************************************************************************
//* Global variables.                                                          *
//******************************************************************************
//Indicates the low frequency SPI mode.
unsigned long isLowFrequencyMode;
//Memory card type.
unsigned char memCardType;
//LBA of the sector in the buffer.
unsigned long bufferedLBA;
//Sector buffer.
unsigned char sectorBuffer[512];


//******************************************************************************
//* Function prototypes of the private functions.                              *
//******************************************************************************
void MemCardSelect();
void MemCardDeselect();
unsigned char MemCardSendCmd(unsigned char, unsigned long);
unsigned char MemCardReadDataBlock(unsigned char *);
unsigned char MemCardWriteDataBlock(unsigned char, unsigned char *);


//******************************************************************************
//* Selects the SD memory card.                                                *
//******************************************************************************
#define SDCARD_LO_FREQ_HZ       250000
#define SDCARD_HI_FREQ_HZ       12500000

void MemCardSelect()
{
    unsigned long clkFreq;

    clkFreq = (isLowFrequencyMode) ? SDCARD_LO_FREQ_HZ : SDCARD_HI_FREQ_HZ;

    SPIDeselect();
    SPIReadWriteByte(0xff);
    SPISelect(SPI_DEVICE_SDCARD, clkFreq);
    SPIReadWriteByte(0xff);
}


//******************************************************************************
//* Deselects the SD memory card.                                              *
//******************************************************************************
void MemCardDeselect()
{
    SPIDeselect();
    SPIReadWriteByte(0xff);
}


//******************************************************************************
//* Sends a command to the memory card.                                        *
//******************************************************************************
unsigned char MemCardSendCmd(unsigned char cmd, unsigned long arg)
{
    unsigned char n = 11;
    unsigned char response = 0xff;

    //Check whether the command is an application specific command.
    if (cmd & 0x80)
    {
        //Send a CMD55 before the application specific command.
        cmd &= 0x7f;
        response = MemCardSendCmd(CMD55, 0);
        //Check the response.
        if (response > 1)
            return response; 
    }

    //Select the memory card.
    MemCardSelect();

    //Initialize the CRC generator.
    SPIInitCRC(SPI_CRC_TYPE_CRC7, SPI_CRC_DIN_MOSI);

    //Send the command to the memory card.
    SPIReadWriteByte(cmd);
    SPIReadWriteByte((unsigned char)(arg >> 24));
    SPIReadWriteByte((unsigned char)(arg >> 16));
    SPIReadWriteByte((unsigned char)(arg >> 8));
    SPIReadWriteByte((unsigned char)arg);
    SPISendCRC(SPI_CRC_TYPE_CRC7);

    //Receive the first byte of the response.
    while ((response & 0x80) && (--n))
        response = SPIReadWriteByte(0xff);

    return response;
}


//******************************************************************************
//* Reads an 512 byte data block from the memory card.                         *
//******************************************************************************
unsigned char MemCardReadDataBlock(unsigned char *dst)
{
    unsigned char data = 0xff;

    //Wait for the data packet (timeout: ~100ms).
    SetTimer(10);
    while ((data == 0xff) && (GetTimer() != 0))
        data = SPIReadWriteByte(0xff);

    //Check the START BLOCK token.
    if (data != 0xfe)
        return DISK_IOERR;
    
    //Initialize the CRC generator.
    SPIInitCRC(SPI_CRC_TYPE_CRC16, SPI_CRC_DIN_MISO);
    //Start the read transfer.
    SPIReadBytes(dst, 512, 0xff);
    
    //Check the CRC value.
    if (SPICheckCRC16() != 0)
        return DISK_IOERR;

    return DISK_OK;
}


//******************************************************************************
//* Writes an 512 byte data block to the memory card.                          *
//******************************************************************************
unsigned char MemCardWriteDataBlock(unsigned char token, unsigned char *src)
{
    unsigned char data = 0;

    //Wait while the card is busy (timeout: ~500ms).
    SetTimer(50);
    while ((data != 0xff) && (GetTimer() != 0))
        data = SPIReadWriteByte(0xff);
    if (data != 0xff)
        return DISK_IOERR;

    //Examine the token.
    if (token == 0)
        return DISK_OK;

    //Send the token to the card.
    SPIReadWriteByte(token);

    //Examine the token.
    if (token != 0xfd)
    {
        //The token is a data token. Initialize the CRC generator.
        SPIInitCRC(SPI_CRC_TYPE_CRC16, SPI_CRC_DIN_MOSI);
        //Start the write transfer.
        SPIWriteBytes(src, 512);
        //Send the CRC value to the card.
        SPISendCRC(SPI_CRC_TYPE_CRC16);

        //Receive and check the response.
        data = SPIReadWriteByte(0xff);
        if ((data & 0x1f) != 0x05)
            return DISK_IOERR;
    }

    return DISK_OK;
}


//******************************************************************************
//* Returns the memory card type.                                              *
//******************************************************************************
unsigned char MemCardGetType()
{
    return memCardType;
}


//******************************************************************************
//* Returns a pointer to the sector buffer.                                    *
//******************************************************************************
unsigned char *MemCardGetSectorBuffer()
{
    return &sectorBuffer[0];
}


//******************************************************************************
//* Initializes the memory card.                                               *
//******************************************************************************
unsigned char MemCardInitialize()
{
    unsigned char  type = 0;
    unsigned char  data[2];

    //Invalidate the content of the sector buffer.
    bufferedLBA = (1 << 31);

    //Select the memory card and set the low frequency mode.
    SPIEnableTransfer();
    MemCardDeselect();
    isLowFrequencyMode = 1;
    
    //Dummy clocks.
    for (data[0] = 50; data[0] > 0; --data[0])
        SPIReadWriteByte(0xff);

    //Reset the memory card.
    if (MemCardSendCmd(CMD0, 0) == 1)
    {
        //Set the high frequency mode.
        isLowFrequencyMode = 0;

        //Enable the CRC check.
        if (MemCardSendCmd(CMD59, 0x0001) == 1)
        {
            //Initialization timeout: ~1000 ms.
            SetTimer(120);

            //Check the memory card type.
            if (MemCardSendCmd(CMD8, 0x01aa) == 1)
            {
                //SD v2. Read the remaining R7 response bytes.
                SPIReadWriteByte(0xff);
                SPIReadWriteByte(0xff);
                data[0] = SPIReadWriteByte(0xff);
                data[1] = SPIReadWriteByte(0xff);

                //Check whether the card can work at 2.7-3.6V.
                if ((data[0] == 0x01) && (data[1] == 0xAA))
                {
                    //Initialize the memory card.
                    while ((GetTimer() != 0) && (MemCardSendCmd(ACMD41, 1UL << 30) != 0));

                    //Check the CCS bit in the OCR register.
                    if ((GetTimer() != 0) && (MemCardSendCmd(CMD58, 0) == 0))
                    {
                        //Read the OCR register.
                        data[0] = SPIReadWriteByte(0xff);
                        SPIReadWriteByte(0xff);
                        SPIReadWriteByte(0xff);
                        SPIReadWriteByte(0xff);

                        //Check whether the memory card is SDHC.
                        type = MEMCARD_TYPE_SD2;
                        if (data[0] & 0x40)
                            type |= MEMCARD_TYPE_BLOCK;
                    }
                }
            }
            else
            {
                //Check the memory card type.
                if (MemCardSendCmd(ACMD41, 0) < 2)
                {
                    //SD v1
                    type = MEMCARD_TYPE_SD1;
                    data[0] = ACMD41;
                }
                else
                {
                    //MMC v3
                    type = MEMCARD_TYPE_MMC;
                    data[0] = CMD1;
                }

                //Initialize the memory card.
                while ((GetTimer() != 0) && (MemCardSendCmd(data[0], 0) != 0));

                //Set the read/write block length to 512 bytes.
                if ((GetTimer() == 0) || (MemCardSendCmd(CMD16, 512) != 0))
                    type = 0;
            }
        }
    }

    //Set the memory card type.
    memCardType = type;

    //Set the high frequency mode and deselect the memory card.
    isLowFrequencyMode = 0;
    MemCardDeselect();

    if (type == 0)
        return DISK_NOINIT;

    return DISK_OK;
}


//******************************************************************************
//* Reads a sector to the sector buffer and returns its (partial) content.     *
//******************************************************************************
unsigned char MemCardReadPartialSector(
    unsigned long  lba,         //LBA of the sector to read.
    unsigned char  *dst,        //Pointer to the read buffer (NULL: no memory copy).
    unsigned short ofs,         //Byte offset to read from (0..511).
    unsigned short len          //Number of bytes to read (ofs+len must be <= 512).
)
{
    unsigned char result = DISK_IOERR;

    //Check the parameters.
    if ((ofs > 511) || ((ofs + len) > 512))
        return DISK_PARERR;

    //Check whether the requested sector is in the buffer.
    if (((bufferedLBA & (1 << 31)) == 0) && (lba == bufferedLBA))
    {
        result = DISK_OK;
        goto no_memcard_read;
    }

    //Convert the LBA to byte address if necessary.
    bufferedLBA = lba;
    if ((memCardType & MEMCARD_TYPE_BLOCK) == 0)
        lba *= 512;

    //Send the CMD17 command (single block read).
    if (MemCardSendCmd(CMD17, lba) == 0)
    {
        //Read a data block from the memory card.
        result = MemCardReadDataBlock(sectorBuffer);

        //Invalidate the sector buffer in case of error.
        if (result != DISK_OK)
            bufferedLBA = (1 << 31);
    }

    //Deselect the memory card.
    MemCardDeselect();

no_memcard_read:
    //Copy the data to the read buffer.
    if ((result == DISK_OK) && (dst != 0))
        memcpy(dst, &sectorBuffer[ofs], len);

    return result;
}


//******************************************************************************
//* Reads multiple sectors from the memory card.                               *
//******************************************************************************
unsigned char MemCardReadMultipleSectors(
    unsigned long lba,         //LBA of the start sector.
    unsigned char *dst,        //Pointer to the read buffer.
    unsigned char count        //Number of sectors to read (1..255).
)
{
    //Check the parameters.
    if (count == 0)
        return DISK_OK;

    //Convert the LBA to byte address if necessary.
    if ((memCardType & MEMCARD_TYPE_BLOCK) == 0)
        lba *= 512;

    //Send the CMD18 command (multiple blocks read).
    if (MemCardSendCmd(CMD18, lba) == 0)
    {
        do
        {
            //Read a data block from the memory card.
            if (MemCardReadDataBlock(dst) != DISK_OK)
                break;

            //Set the new location in the read buffer.
            dst += 512;
        } while (--count);

        //Send the CMD12 command (transfer stop).
        MemCardSendCmd(CMD12, 0);
    }

    //Deselect the memory card.
    MemCardDeselect();

    if (count > 0)
        return DISK_IOERR;

    return DISK_OK;
}


//******************************************************************************
//* Writes data to the sector buffer, starts a single sector write operation.  *
//******************************************************************************
unsigned char MemCardWritePartialSector(
    unsigned long  lba,         //LBA of the sector to write (used only when src is NULL).
    unsigned char  *src,        //Pointer to the data to be written (NULL: start write).
    unsigned short ofs,         //Byte offset to write to (0..511).
    unsigned short len          //Number of bytes to write (ofs+len must be <= 512).
)
{
    unsigned char result = DISK_IOERR;

    //Check the parameters.
    if ((ofs > 511) || ((ofs + len) > 512))
        return DISK_PARERR;

    //Check whether the data in the sector buffer is valid.
    if ((unsigned char)(bufferedLBA >> 24) & 0x80)
        return DISK_PARERR;

    //Write the data into the sector buffer if necessary.
    if (src)
    {
        if (len != 0)
            memcpy(&sectorBuffer[ofs], src, len);
        return DISK_OK;
    }

    //Convert the LBA to byte address if necessary.
    bufferedLBA = lba;
    if ((memCardType & MEMCARD_TYPE_BLOCK) == 0)
        lba *= 512;

    //Send the CMD14 command (single block write).
    if (MemCardSendCmd(CMD17, lba) == 0)
    {
        //Write a data block to the memory card.
        result = MemCardWriteDataBlock(0xfe, sectorBuffer);
    }

    //Wait while the card is busy.
    MemCardWriteDataBlock(0, 0);

    //Deselect the memory card.
    MemCardDeselect();

    return result;
}


//******************************************************************************
//* Writes multiple sectors to the memory card.                                *
//******************************************************************************
unsigned char MemCardWriteMultipleSectors(
    unsigned long lba,         //LBA of the start sector.
    unsigned char *src,        //Pointer to the data to be written.
    unsigned char count        //Number of sectors to write (1..255).
)
{
    //Check the parameters.
    if (count == 0)
        return DISK_OK;

    //Convert the LBA to byte address if necessary.
    if ((memCardType & MEMCARD_TYPE_BLOCK) == 0)
        lba *= 512;

    //Send the number of blocks to be pre-erased (ACMD23 command).
    if (memCardType & MEMCARD_TYPE_SDC)
        MemCardSendCmd(ACMD23, count);

    //Send the CMD25 command (multiple blocks write).
    if (MemCardSendCmd(CMD25, lba) == 0)
    {
        do
        {
            //Write a data block to the memory card.
            if (MemCardWriteDataBlock(0xfc, src) != DISK_OK)
                break;

            //Set the new location in the buffer.
            src += 512;
        } while (--count);

        //Send the STOP TRAN token.
        if (MemCardWriteDataBlock(0xfd, 0) != DISK_OK)
            count = 1;
    }

    //Wait while the card is busy.
    MemCardWriteDataBlock(0, 0);

    //Deselect the memory card.
    MemCardDeselect();

    if (count > 0)
        return DISK_IOERR;

    return DISK_OK;
}

