//******************************************************************************
//* ENC424J600 handler routines.                                               *
//* Written by   : Tamas Raikovich (rtamas@mit.bme.hu)                         *
//* Version      : 1.0                                                         *
//* Last modified: 06/24/2010                                                  *
//******************************************************************************
#include "xparameters.h"
#include "enc424j600_registers.h"
#include "enc424j600.h"
#include "ethernet_interface.h"


//*****************************************************************************
//* Writes the Ethernet PHY register.                                         *
//*****************************************************************************
unsigned long EncWritePhyRegister(unsigned long mem_baseaddr, unsigned long phyreg, unsigned short value)
{
	//Set the Ethernet PHY register address.
	EncWriteReg16(mem_baseaddr, MIREGADR, 0x0100 | phyreg);
	//Write the data to the PHY register.
	EncWriteReg16(mem_baseaddr, MIWR, value);
	//Wait for the write to complete.
	while (EncReadReg8(mem_baseaddr, MISTATL) & MISTATL_BUSY);
	
	return ETH_STATUS_SUCCESS;
}


//*****************************************************************************
//* Initializes the ENC424J600.                                               *
//*****************************************************************************
unsigned long EncInitialize(unsigned long mem_baseaddr)
{
	volatile unsigned long i;

	//Wait while the PSP interface cannot be used.
	for (;;)
	{
		EncWriteReg16(mem_baseaddr, EUDAST, 0x1234);
		if (EncReadReg16(mem_baseaddr, EUDAST) == 0x1234)
			break;
	}

	//Wait for the CLKRDY bit.
    while ((EncReadReg8(mem_baseaddr, ESTATH) & ESTATH_CLKRDY) == 0);
	
	//Reset command.
	EncWriteReg8(mem_baseaddr, ECON2L, ECON2L_ETHRST);
	
	//Wait.
	for (i = 0; i < 25 * (XPAR_PROC_BUS_0_FREQ_HZ / 1000000); i++);
	
	//Check the value of the EUDAST register.
	if (EncReadReg16(mem_baseaddr, EUDAST) != 0x0000)
		return ETH_STATUS_ERROR;
		
	//Wait.
	for (i = 0; i < 256 * (XPAR_PROC_BUS_0_FREQ_HZ / 1000000); i++);

	return ETH_STATUS_SUCCESS;
}


//*****************************************************************************
//* Sets the RX buffer size in the ENC424J600.                                *
//*****************************************************************************
unsigned long EncSetRxBufferSize(unsigned long mem_baseaddr, unsigned long size, unsigned long *nextRxPacketPtr)
{
	//Check the parameters.
	if ((size & 0x01) || (size > 0x5a00))
		return ETH_STATUS_ERROR;
		
	//Determine the start address of the RX buffer.
	size = 0x6000 - size;
	*nextRxPacketPtr = size;
	
	//Set the start address of the RX buffer.
	EncWriteReg16(mem_baseaddr, ERXST, size);
	EncWriteReg16(mem_baseaddr, ERXTAIL, 0x5fff);
	
	return ETH_STATUS_SUCCESS;
}


//*****************************************************************************
//* Reads the MAC address from the ENC424J600.                                *
//*****************************************************************************
unsigned long EncReadMacAddress(unsigned long mem_baseaddr, unsigned char *macaddr)
{
	//Check the parameters.
	if (macaddr == 0)
		return ETH_STATUS_ERROR;
	
	//Read the MAC address.
	macaddr[0] = (unsigned char)EncReadReg8(mem_baseaddr, MAADR1L);
	macaddr[1] = (unsigned char)EncReadReg8(mem_baseaddr, MAADR1H);
	macaddr[2] = (unsigned char)EncReadReg8(mem_baseaddr, MAADR2L);
	macaddr[3] = (unsigned char)EncReadReg8(mem_baseaddr, MAADR2H);
	macaddr[4] = (unsigned char)EncReadReg8(mem_baseaddr, MAADR3L);
	macaddr[5] = (unsigned char)EncReadReg8(mem_baseaddr, MAADR3H);
	
	return ETH_STATUS_SUCCESS;	
}


//*****************************************************************************
//* Initializes the Ethernet MAC and enables the packet reception.            *
//*****************************************************************************
unsigned long EncInitializeMac(unsigned long mem_baseaddr, unsigned char *macaddr, unsigned long maxframelen)
{
	//Check the parameters.
	if ((maxframelen < 64) && (maxframelen > 1518))
		return ETH_STATUS_ERROR;
	
	//Set the MAC address if necessary.
	if (macaddr != 0)
	{
		EncWriteReg8(mem_baseaddr, MAADR1L, macaddr[0]);
		EncWriteReg8(mem_baseaddr, MAADR1H, macaddr[1]);
		EncWriteReg8(mem_baseaddr, MAADR2L, macaddr[2]);
		EncWriteReg8(mem_baseaddr, MAADR2H, macaddr[3]);
		EncWriteReg8(mem_baseaddr, MAADR3L, macaddr[4]);
		EncWriteReg8(mem_baseaddr, MAADR3H, macaddr[5]);
	}
	
	//Set the max. frame length.
	EncWriteReg16(mem_baseaddr, MAMXFL, maxframelen);
	
	//Configure the Ethernet PHY.
	EncWritePhyRegister(mem_baseaddr, PHANA, PHANA_ADPAUS0 | PHANA_AD10FD | PHANA_AD10 | PHANA_AD100FD | PHANA_AD100 | PHANA_ADIEEE0);
	
	//Enable the packet reception.
	EncSetRegBits8(mem_baseaddr, ECON1L, ECON1L_RXEN);
		
	return ETH_STATUS_SUCCESS;
}


//*****************************************************************************
//* Receives a packet.                                                        *
//*****************************************************************************
unsigned long EncReceivePacket(unsigned long mem_baseaddr, void *dst, unsigned short *len, unsigned long *nextRxPacketPtr)
{
	unsigned long status = ETH_STATUS_SUCCESS;
	unsigned char rxstatus[8];
	
	//Check the parameters.
	if ((dst == 0) || (len == 0) || (nextRxPacketPtr == 0))
		return ETH_STATUS_ERROR;
		
	//Check whether the RX buffer contains valid data.
	if ((EncReadReg8(mem_baseaddr, EIRL) & EIRL_PKTIF) == 0)
	{
		*len = 0;
		return ETH_STATUS_SUCCESS;
	}
	
	//Set the start address of the packet.
	EncWriteReg16(mem_baseaddr, ERXRDPT, (unsigned short)(*nextRxPacketPtr));
	
	//Read the status information.
	EncCopyFromSram(mem_baseaddr, ERXDATA, rxstatus, sizeof(rxstatus));
	
	//Determine the start address of the next packet.
	*nextRxPacketPtr = (unsigned long)rxstatus[0] + ((unsigned long)rxstatus[1] << 8);
	
	//Determine the packet length.
	*len = (unsigned short)rxstatus[2] + ((unsigned short)rxstatus[3] << 8);
	
	//Check whether RX error occurred.
	if (rxstatus[4] & 0x80)
	{
		//Read the data from the RX buffer.
		EncCopyFromSram(mem_baseaddr, ERXDATA, dst, *len);
	}
	else
	{
		//RX error.
		*len = 0;
		status = ETH_STATUS_ERROR;
	}
	
	//Set the new end address of the RX buffer.
	EncWriteReg16(mem_baseaddr, ERXTAIL, (unsigned short)(*nextRxPacketPtr) - 2);
	
	//Acknowledge the received packet.
	EncSetRegBits8(mem_baseaddr, ECON1H, ECON1H_PKTDEC);
	
	return status;
}


//*****************************************************************************
//* Sends the packet in the TX buffer.                                        *
//*****************************************************************************
unsigned long EncWaitForTxComplete(unsigned long mem_baseaddr)
{
	//Wait for the transfer to complete.
	while ((EncReadReg8(mem_baseaddr, ECON1L) & ECON1L_TXRTS) != 0);
	//Clear the interrupt flag.
	EncClearRegBits8(mem_baseaddr, EIRL, EIRL_TXIF | EIRL_TXABTIF);
	
	return ETH_STATUS_SUCCESS;
}

unsigned long EncTransmitPacket(unsigned long mem_baseaddr, unsigned long txstartaddr, unsigned short len)
{
	//Set the start address of the packet.
	EncWriteReg16(mem_baseaddr, ETXST, txstartaddr);
	//Set the packet length.
	EncWriteReg16(mem_baseaddr, ETXLEN, len);
	//Send the packet.
	EncSetRegBits8(mem_baseaddr, ECON1L, ECON1L_TXRTS);
	
	return ETH_STATUS_SUCCESS;
}
