//******************************************************************************
//* ENC424J600 data transfer functions.                                        *
//* Written by   : Tamas Raikovich (rtamas@mit.bme.hu)                         *
//* Version      : 1.0                                                         *
//* Last modified: 06/24/2010                                                  *
//******************************************************************************
#include "xio.h"
#include "enc424j600_registers.h"
#include "ethernet_interface.h"


//*****************************************************************************
//* 16 bit SFR operations.                                                   *
//*****************************************************************************
void EncWriteReg16(unsigned long mem_baseaddr, unsigned long reg, unsigned short value)
{
	//Write the lower byte of the register.
    EncWriteReg8(mem_baseaddr, reg, (unsigned char)value);
	//Write the upper byte of the register.
    EncWriteReg8(mem_baseaddr, reg + (1 << REG_ADDR_SHIFT), (unsigned char)(value >> 8));
}

unsigned short EncReadReg16(unsigned long mem_baseaddr, unsigned long reg)
{
	unsigned short l, h;
	
	//Read the lower byte of the register.
	l = (unsigned short)EncReadReg8(mem_baseaddr, reg);
	//Read the upper byte of the register.
	h = (unsigned short)EncReadReg8(mem_baseaddr, reg + (1 << REG_ADDR_SHIFT));
	
	return l + (h << 8);
}

void EncSetRegBits16(unsigned long mem_baseaddr, unsigned long reg, unsigned short mask)
{
    //Write the lower byte of the register.
    EncSetRegBits8(mem_baseaddr, reg, (unsigned char)mask);
    //Write the upper byte of the register.
    EncSetRegBits8(mem_baseaddr, reg + (1 << REG_ADDR_SHIFT), (unsigned char)(mask >> 8));
}

void EncClearRegBits16(unsigned long mem_baseaddr, unsigned long reg, unsigned short mask)
{
    //Write the lower byte of the register.
    EncClearRegBits8(mem_baseaddr, reg, (unsigned char)mask);
    //Write the upper byte of the register.
    EncClearRegBits8(mem_baseaddr, reg + (1 << REG_ADDR_SHIFT), (unsigned char)(mask >> 8));
}


//*****************************************************************************
//* Reads/writes the ENC424J600 SRAM buffers.                                 *
//*****************************************************************************
void EncCopyToSram(unsigned long mem_baseaddr, unsigned long buffer, void *src, unsigned long len)
{
    unsigned char *ptr = (unsigned char *)src;
	unsigned long reg_addr = mem_baseaddr + (buffer & REG_ADDR_MASK);
    unsigned long count;
	
	//Copy bytes to reach the word boundary.
	while (((unsigned long)ptr & 0x03) && (len > 0))
	{
		XIo_Out8(reg_addr, *ptr);
		ptr++;
		len--;
	}
    
    //Determine the number of 32-bit words to copy.
    count = len >> 2;
    len &= 0x03;
    //Copy the data.
	while (count > 0)
	{
		XIo_Out32(reg_addr, *(unsigned long *)ptr);
		ptr += 4;
		count--;
	}

    //Copy the remaining bytes.
	while (len > 0)
	{
		XIo_Out8(reg_addr, *ptr);
		ptr++;
		len--;
	}
}

void EncCopyFromSram(unsigned long mem_baseaddr, unsigned long buffer, void *dst, unsigned long len)
{
    unsigned char *ptr = (unsigned char *)dst;
	unsigned long reg_addr = mem_baseaddr + (buffer & REG_ADDR_MASK);
    unsigned long count;
	
    //Copy bytes to reach the word boundary.
	while (((unsigned long)ptr & 0x03) && (len > 0))
	{
		*ptr = XIo_In8(reg_addr);
		ptr++;
		len--;
	}
    
	//Determine the number of 32-bit words to copy.
    count = len >> 2;
    len &= 0x03;
    //Copy the data.
	while (count > 0)
	{
		*(unsigned long *)ptr = XIo_In32(reg_addr);
		ptr += 4;
		count--;
	}

	//Copy the remaining bytes.
	while (len > 0)
	{
		*ptr = XIo_In8(reg_addr);
		ptr++;
		len--;
	}
}
