//******************************************************************************
//* FAT filesystem driver.                                                     *
//* Written by   : Tamas Raikovich                                             *
//* Version      : 1.0                                                         *
//* Last modified: 2012.10.11.                                                 *
//******************************************************************************
//******************************************************************************
//*                                                                            *
//* This file is based on the Petit FatFs by ChaN (http://elm-chan.org).       *
//*                                                                            *
//* Petit FatFs module is an open source software to implement FAT file system *
//* to small embedded systems. This is a free software and is opened for       *
//* education, research and commercial developments under license policy of    *
//* following trems.                                                           *
//*                                                                            *
//* Copyright (C) 2009, ChaN, all right reserved.                              *
//*                                                                            *
//* - The Petit FatFs module is a free software and there is NO WARRANTY.      *
//* - No restriction on use. You can use, modify and redistribute it for       *
//*   personal, non-profit or commercial use UNDER YOUR RESPONSIBILITY.        *
//* - Redistributions of source code must retain the above copyright notice.   *
//******************************************************************************
#include <string.h>
#include "fat_filesystem.h"
#include "memory_card.h"


//******************************************************************************
//* Multi-byte word access functions.                                          *
//******************************************************************************
unsigned short LD_WORD(void *addr)
{
    unsigned char *ptr = (unsigned char *)addr;
    unsigned short value = 0;

    value |= (unsigned short)*ptr++;
    value |= (unsigned short)*ptr << 8;

    return value;
}

unsigned long LD_DWORD(void *addr)
{
    unsigned char *ptr = (unsigned char *)addr;
    unsigned long value = 0;

    value |= (unsigned long)*ptr++;
    value |= ((unsigned long)*ptr++) << 8;
    value |= ((unsigned long)*ptr++) << 16;
    value |= ((unsigned long)*ptr) << 24;

    return value;
}


//******************************************************************************
//* Global variables.                                                          *
//******************************************************************************
//Pointer to the file system object.
PFATFS  FatFs;


//******************************************************************************
//* Function prototypes of the private functions.                              *
//******************************************************************************
unsigned long get_fat(unsigned long);
unsigned long clust2sect(unsigned long);
unsigned char dir_rewind(PDIR);
unsigned char dir_next(PDIR);
unsigned char dir_find(PDIR);
unsigned char dir_read(PDIR);
unsigned char create_name(PDIR, const char **);
void get_fileinfo(PDIR, PFILINFO);
unsigned char follow_path(PDIR, const char *);
unsigned char check_fs(unsigned long);
unsigned char is_eoc(unsigned long);


//******************************************************************************
//* Reads the value of a FAT entry.                                            *
//******************************************************************************
unsigned long get_fat(unsigned long clst)
{
    unsigned short wc, bc, ofs;
    unsigned char  buf[4];
    PFATFS         fs = FatFs;

    //Range check.
    if ((clst < 2) || (clst >= fs->max_clust))
        return 1;

    //Examine the FAT type.
    switch (fs->fs_type)
    {
        case FS_FAT12:
            bc = (unsigned short) clst;
            bc += bc / 2;
            ofs = bc % 512;
            bc /= 512;
            if (ofs != 511)
            {
                if (DiskReadPartialSector(fs->fatbase + bc, buf, ofs, 2))
                    break;
            } 
            else 
            {
                if (DiskReadPartialSector(fs->fatbase + bc, buf, 511, 1))
                    break;
                if (DiskReadPartialSector(fs->fatbase + bc + 1, buf + 1, 0, 1))
                    break;
            }
            wc = LD_WORD(buf);
            if (clst & 1)
                return wc >> 4;
            return wc & 0x0fff;
        
        case FS_FAT16:
            if (DiskReadPartialSector(fs->fatbase + (clst / 256), buf, ((unsigned short)clst % 256) * 2, 2))
                break;
            return LD_WORD(buf);
        
        case FS_FAT32:
            if (DiskReadPartialSector(fs->fatbase + (clst / 128), buf, ((unsigned short)clst % 128) * 4, 4))
                break;
            return LD_DWORD(buf) & 0x0fffffff;
    }

    return 1;
}


//******************************************************************************
//* Returns the first sector of the given cluster.                             *
//******************************************************************************
unsigned long clust2sect(unsigned long clst)
{
    PFATFS fs = FatFs;
    
    //Check whether the cluster number is valid.
    if (clst >= fs->max_clust)
        return 0;

    return ((clst - 2) << fs->log2csize) + fs->database;
}


//******************************************************************************
//* Directory handling: rewind directory index.                                *
//******************************************************************************
unsigned char dir_rewind(PDIR dj)
{
    unsigned long clst = dj->sclust;
    PFATFS        fs   = FatFs;

    dj->index = 0;

    //Check the start cluster range.
    if ((clst == 1) || (clst >= fs->max_clust))
        return FR_DISK_ERR;

    //Replace cluster number 0 with root cluster number in case of FAT32.
    if ((clst == 0) && (fs->fs_type == FS_FAT32))
        clst = fs->dirbase;

    //Current cluster and sector.
    dj->clust = clst;
    dj->sect = (clst != 0) ? clust2sect(clst) : fs->dirbase;

    return FR_OK;
}


//******************************************************************************
//* Directory handling: move to the next directory index.                      *
//******************************************************************************
unsigned char dir_next(PDIR dj)
{
    unsigned long  clst;
    unsigned short i  = dj->index + 1;
    PFATFS         fs = FatFs;
    
    //Report EOT when index has reached 65535
    if ((i == 0) || (dj->sect == 0))
        return FR_NO_FILE;

    //Check whether the sector has been changed.
    if ((i & 0x0f) == 0)
    {
        //Next sector.
        dj->sect++;

        //Check the table type.
        if (dj->clust == 0)
        {
            //Static table. Report EOT when end of table.
            if (i >= fs->n_rootdir)
                return FR_NO_FILE;
        }
        else 
        {
            //Dynamic table. Check whether the cluster number has been changed.
            if (((i / 16) & (fs->csize - 1)) == 0)
            {
                //The cluster number has been changed. Get the next cluster.
		clst = get_fat(dj->clust);
		if (clst <= 1)
                    return FR_DISK_ERR;

                //Report EOT when the end of the dynamic table has been reached.
		if (clst >= fs->max_clust)
                    return FR_NO_FILE;

                //Current cluster and sector.
		dj->clust = clst;
		dj->sect = clust2sect(clst);
	    }
        }
    }

    dj->index = i;

    return FR_OK;
}


//******************************************************************************
//* Directory handling: find an object in the directory.                       *
//******************************************************************************
unsigned char dir_find(PDIR dj)
{
    unsigned char        res;
    PFAT_DIR_ENTRY dir = (PFAT_DIR_ENTRY)FatFs->buf;

    //Rewind the directory object.
    res = dir_rewind(dj);
    if (res != FR_OK)
        return res;

    do
    {
        res = FR_OK;
        //Read a directory entry.
        if (DiskReadPartialSector(dj->sect, (void *)dir, (dj->index % 16) * 32, 32) != DISK_OK)
        {
            res = FR_DISK_ERR;
            break;
        }

        //Check whether the end of the table has been reached.
        if (dir->DIR_Name[0] == 0)
        {
            //The end of the table has been reached.
            res = FR_NO_FILE;
            break;
        }

        //Check whether the directory entry is valid.
        if (((dir->DIR_Attr & AM_VOL) == 0) && (memcmp(dir, dj->fn, 11) == 0))
            break;

        //Move the pointer to the next directory entry.
        res = dir_next(dj);
    } while (res == FR_OK);

    return res;
}


//******************************************************************************
//* Reads the next directory entry.                                            *
//******************************************************************************
#if FATFS_USE_DIR

unsigned char dir_read(PDIR dj)
{
    unsigned char  c;
    unsigned char  res = FR_NO_FILE;
    PFAT_DIR_ENTRY dir = (PFAT_DIR_ENTRY)FatFs->buf;

    while (dj->sect)
    {
        //Read a directory entry.
        res = DiskReadPartialSector(dj->sect, (void *)dir, (dj->index % 16) * 32, 32) ? FR_DISK_ERR : FR_OK;
        if (res != FR_OK)
            break;

        //Check whether the end of the table has been reached.
        c = dir->DIR_Name[0];
        if (c == 0)
        {
            //The end of the table has been reached.
            res = FR_NO_FILE;
            break;
        }
        
        //Check whether the directory entry is valid.
        if ((c != 0xe5) && (c != '.') && ((dir->DIR_Attr & AM_VOL) == 0))
            break;
        
        //Move the pointer to the next directory entry.
        res = dir_next(dj);
        if (res != FR_OK)
            break;
    }

    if (res != FR_OK)
        dj->sect = 0;

    return res;
}

#endif //FATFS_USE_DIR


//******************************************************************************
//* Returns the next path segment.                                             *
//******************************************************************************
unsigned char create_name(PDIR dj, const char **path)
{
    unsigned char c;
    unsigned char ni   = 8;
    unsigned char si   = 0;
    unsigned char i    = 0;
    unsigned char *sfn = dj->fn;
    const char    *p   = *path;

    //Fill the file name with spaces.
    memset(sfn, ' ', 11);

    //Create file name in directory form.
    for (;;)
    {
        //Read the next character from the path.
        c = p[si++];

        //Break on end of segment.
        if ((c <= ' ') || (c == '/'))
            break;
        if ((c == '.') || (i >= ni))
        {
            if ((ni != 8) || (c != '.'))
                break;
            i = 8;
            ni = 11;
            continue;
        }

        //Convert the character to upper-case.
        if ((c >= 'a') && (c <= 'z'))
            c -= 0x20;
        
        //Append the character to the file name.
        sfn[i++] = c;
    }

    //Set the pointer to the next segment.
    *path = &p[si];
    //Set the last segment flag if end of path.
    sfn[11] = (c <= ' ') ? 1 : 0;

    return FR_OK;
}


//******************************************************************************
//* Reads the file information from the directory entry.                       *
//******************************************************************************
#if FATFS_USE_DIR

void get_fileinfo(PDIR dj, PFILINFO fno)
{
    unsigned char  c;
    unsigned char  i;
    PFAT_DIR_ENTRY dir = (PFAT_DIR_ENTRY)FatFs->buf;
    unsigned char  *p  = fno->fname;

    if (dj->sect) 
    {
        //Copy the file name.
        for (i = 0; i < 8; i++)
        {
            c = dir->DIR_Name[i];
            if (c == ' ')
                break;
            if (c == 0x05)
                c = 0xe5;
            *p++ = c;
        }

        //Copy the file extension.
        if (dir->DIR_Name[8] != ' ')
        {
            *p++ = '.';
            for (i = 8; i < 11; i++)
            {
                c = dir->DIR_Name[i];
                if (c == ' ')
                    break;
                *p++ = c;
            }
        }

        //Copy the attribute, size, date and time.
        fno->fattrib = dir->DIR_Attr;
        fno->fsize = LD_DWORD(&dir->DIR_FileSize);
        fno->fdate = LD_WORD(&dir->DIR_WrtDate);
        fno->ftime = LD_WORD(&dir->DIR_WrtTime);
    }

    //Set the zero termination.
    *p = 0;
}

#endif //FATFS_USE_DIR


//******************************************************************************
//* Follows a file path.                                                       *
//******************************************************************************
unsigned char follow_path(PDIR dj, const char *path)
{
    PFAT_DIR_ENTRY dir;
    unsigned char  res;

    //Skip the leading spaces.
    while (*path == ' ')
        path++;
    //Skip the heading separator.
    if (*path == '/')
        path++;
    //Set the root directory as start directory.
    dj->sclust = 0;

    //Check the first character of the path.
    if ((unsigned char)*path <= ' ')
    {
        //Null path means the root directory.
        res = dir_rewind(dj);
        FatFs->buf[0] = 0;
    }
    else 
    {
        //Follow the path.
        for (;;)
        {
            //Get the next path segment.
            res = create_name(dj, &path);
            if (res != FR_OK)
                break;
            
            //Find the path segment in the directory.
            res = dir_find(dj);
            if (res != FR_OK) 
            {
                //Could not find the directory entry.
                if ((res == FR_NO_FILE) && (dj->fn[11] == 0))
                    res = FR_NO_PATH;
                break;
            }
            
            //Check whether it is the last segment.
            if (dj->fn[11] != 0)
                break;

            //It is not the last segment. Follow the sub-directory.
            dir = (PFAT_DIR_ENTRY)FatFs->buf;
            if ((dir->DIR_Attr & AM_DIR) == 0)
            {
                //Cannot follow because it is a file.
                res = FR_NO_PATH;
                break;
            }

            dj->sclust = ((unsigned long)LD_WORD(&dir->DIR_FstClusHI) << 16) | LD_WORD(&dir->DIR_FstClusLO);
        }
    }

    return res;
}


//******************************************************************************
//* Checks a sector if it is an FAT boot record.                               *
//******************************************************************************
unsigned char check_fs(unsigned long lba)
{
    PBOOT_SECTOR  buff;
    unsigned char *ptr;

    //Read the sector to the sector buffer.
    if (DiskReadPartialSector(lba, 0, 0, 512) != DISK_OK)
    {
        //Indicate the error.
        return 3;
    }

    //Check the boot record signature.
    buff = (PBOOT_SECTOR)DiskGetSectorBuffer();
    if (LD_WORD(&buff->BS_SignatureWord) != 0xaa55)
    {
        //Not a boot record.
        return 2;
    }

    //Check FAT12/16.
    ptr = &buff->fatSpecificData.fat12Fat16Data.BS_FilSysType[0];
    if ((ptr[0] == 'F') && (ptr[1] == 'A') && (ptr[2] == 'T'))
    {
        //FAT boot record.
        return 0;
    }

    //Check FAT32.
    ptr = &buff->fatSpecificData.fat32Data.BS_FilSysType[0];
    if ((ptr[0] == 'F') && (ptr[1] == 'A') && (ptr[2] == 'T'))
    {
        //FAT boot record.
        return 0;
    }

    //Valid boot record but not an FAT.
    return 1;
}


//******************************************************************************
//* Checks whether the current FAT entry is the EOC mark.                      *
//******************************************************************************
unsigned char is_eoc(unsigned long clust)
{
    //Zero-length file has a first cluster number of 0 in its directory entry.
    if (clust == 0)
        return 1;

    //Examine the FAT type.
    switch (FatFs->fs_type)
    {
        case FS_FAT12:
            return (clust >= 0x00000ff8UL);     //FAT12
        case FS_FAT16:
            return (clust >= 0x0000fff8UL);     //FAT16
        default:
            return (clust >= 0x0ffffff8UL);     //FAT32
    }
}


//******************************************************************************
//* Mounts or unmounts a logical drive.                                        *
//******************************************************************************
unsigned char pf_mount(PFATFS fs)
{
    unsigned long fsize, tsect, tmp_clst, rootdir_sect;
    PBOOT_SECTOR  buff;
    unsigned char fmt;
    unsigned long bsect = 0;

    //Unregister the filesystem object.
    FatFs = 0;
    if (fs == 0)
        return FR_OK;

    //Check if the drive is ready or not.
    if (DiskInitialize() != DISK_OK)
        return FR_NOT_READY;

    //Search for a FAT partition on the drive. Check sector 0.
    fmt = check_fs(bsect);
    if (fmt == 1)
    {
        //Not an FAT boot record, it may be FDISK format.
        //Check the first partition listed in the partition table.
        if (DiskReadPartialSector(bsect, 0, 0, 512) != DISK_OK)
        {
            fmt = 3;
        }
        else
        {
            //Check if the partition is existing.
            buff = (PBOOT_SECTOR)DiskGetSectorBuffer();
            if (buff->partitionData[0].systemID != 0)
            {
                //Get the first sector number of the partition.
                bsect = LD_DWORD(&buff->partitionData[0].relativeSector);
                //Check the boot record of the partition.
                fmt = check_fs(bsect);
            }
        }
    }

    if (fmt == 3)
        return FR_DISK_ERR;
    if (fmt != 0)
        return FR_NO_FILESYSTEM;

    //Read the boot record of the partition.
    if (DiskReadPartialSector(bsect, 0, 0, 512) != DISK_OK)
        return FR_DISK_ERR;
    buff = (PBOOT_SECTOR)DiskGetSectorBuffer();

    //Check the number of bytes per sector (only 512 is supported).
    if (LD_WORD(&buff->BPB_BytsPerSec) != 512)
        return FR_NO_FILESYSTEM;

    //Calculate the size of the FAT.
    fsize = LD_WORD(&buff->BPB_FATSz16);
    if (fsize == 0)
        fsize = LD_DWORD(&buff->fatSpecificData.fat32Data.BPB_FATSz32);
    tmp_clst = fsize;
    for (fmt = 1; fmt < buff->BPB_NumFATs; ++fmt)
        fsize += tmp_clst;

    //FAT start sector (LBA).
    fs->fatbase = bsect + LD_WORD(&buff->BPB_RsvdSecCnt);

    //Number of sectors per cluster (must be power of 2).
    fmt = buff->BPB_SecPerClus;
    fs->csize = buff->BPB_SecPerClus;
    fs->log2csize = 0;
    while ((fmt & 0x01) == 0)
    {
        fmt >>= 1;
        fs->log2csize++;
    }
    if (fmt != 0x01)
        return FR_NO_FILESYSTEM;

    //Total number of sectors on the volume.
    tsect = LD_WORD(&buff->BPB_TotSec16);
    if (tsect == 0)
        tsect = LD_DWORD(&buff->BPB_TotSec32);

    //Number of root directory entries.
    fs->n_rootdir = LD_WORD(&buff->BPB_RootEntCnt);
    rootdir_sect = ((unsigned long)fs->n_rootdir * 32 + 511) / 512;

    //Number of clusters on the volume.
    tmp_clst = (tsect - (LD_WORD(&buff->BPB_RsvdSecCnt) + fsize + rootdir_sect)) >> fs->log2csize;
    fs->max_clust = tmp_clst + 2;

    //Determine the FAT type.
    fmt = FS_FAT12;
    if (tmp_clst >= 4085UL)
        fmt = FS_FAT16;
    if (tmp_clst >= 65525UL)
        fmt = FS_FAT32;
    fs->fs_type = fmt;

    //Root directory start cluster (FAT32) or sector (FAT12/16).
    if (fmt == FS_FAT32)
        fs->dirbase = LD_DWORD(&buff->fatSpecificData.fat32Data.BPB_RootClus);
    else
        fs->dirbase = fs->fatbase + fsize;
    
    //First data sector.
    fs->database = fs->fatbase + fsize + rootdir_sect;

    fs->flag = 0;
    FatFs = fs;

    return FR_OK;
}


//******************************************************************************
//* Opens a file.                                                              *
//******************************************************************************
unsigned char pf_open(const char *path)
{
    unsigned char res;
    DIR           dj;
    unsigned char buff[12];
    FAT_DIR_ENTRY dir;
    PFATFS        fs = FatFs;

    //check if the volume is mounted.
    if (fs == 0)
        return FR_NOT_ENABLED;

    fs->flag = 0;
    fs->buf = (unsigned char *)&dir;
    dj.fn = buff;

    //Follow the file path.
    res = follow_path(&dj, path);
    if (res != FR_OK)
        return res;

    //Check whether the directory entry is a valid file.
    if ((dir.DIR_Name[0] == 0) || (dir.DIR_Attr & AM_DIR))
        return FR_NO_FILE;

    //First cluster of the file.
    fs->org_clust =	((unsigned long)LD_WORD(&dir.DIR_FstClusHI) << 16) | LD_WORD(&dir.DIR_FstClusLO);
    fs->curr_clust = fs->org_clust;
    //File size.
    fs->fsize = LD_DWORD(&dir.DIR_FileSize);
    //Check if it is a zero-length file.
    if (is_eoc(fs->curr_clust) || (fs->fsize == 0))
        return FR_NO_FILE;
    //Current data sector.
	fs->dsect = clust2sect(fs->curr_clust);
    //Sector offset in the cluster.
	fs->csect = 0;
    //File pointer.
    fs->fptr = 0;
    fs->flag = FA_OPENED;

    return FR_OK;
}


//******************************************************************************
//* Read data from the opened file.                                            *
//******************************************************************************
#if FATFS_USE_READ

unsigned char pf_read(
    void           *buff,   //Pointer to the read buffer.
    unsigned short btr,     //Number of bytes to read.
    unsigned short *br      //Pointer to number of bytes read.
)
{
    unsigned short bcnt, bofs, scnt, sofs;
    unsigned long  temp;
    unsigned char  *rdbuff = buff;
    PFATFS         fs = FatFs;

    //Clear the number of bytes read.
    *br = 0;
    
    //Check if the volume is mounted.
    if (fs == 0)
        return FR_NOT_ENABLED;
    //Check if the file is opened.
    if ((fs->flag & FA_OPENED) == 0)
        return FR_NOT_OPENED;

    //Calculate the remaining bytes.
    temp = fs->fsize - fs->fptr;
    if ((unsigned long)btr > temp)
        btr = (unsigned short)temp;

    //Read the data.
    while (btr > 0)
    {
        //Check whether the file pointer is at sector boundary.
        bofs = (unsigned short)fs->fptr % 512;
        if ((bofs > 0) || (btr < 512))
        {
            //Calculate the number of bytes to read.
            bcnt = 512 - bofs;
            if (btr < bcnt)
                bcnt = btr;
            //Increment the sector offset when the sector boundary is reached.
            scnt = (bofs + bcnt) / 512;
            //Read a partial sector.
            if (DiskReadPartialSector(fs->dsect, rdbuff, bofs, bcnt) != DISK_OK)
                goto fr_abort;
        }
        else
        {
            //Calculate the number of sectors to read.
            sofs = ((unsigned short)fs->fptr / 512) & (unsigned short)(fs->csize - 1);
            scnt = (unsigned short)fs->csize - sofs;
            sofs = btr / 512;
            if (scnt > 127)
                scnt = 127;
            if (scnt > sofs)
                scnt = sofs;
            //Number of bytes to read.
            bcnt = scnt * 512;
            //Read whole sectors.
            if (MemCardReadMultipleSectors(fs->dsect, rdbuff, (unsigned char)scnt) != DISK_OK)
                goto fr_abort;
        }

        //Update the pointers.
        fs->csect += scnt;
        fs->dsect += scnt;
        fs->fptr += bcnt;
        rdbuff += bcnt;
        btr -= bcnt;
        *br += bcnt;

        //Check whether the cluster boundary has been reached.
        if (fs->csect >= fs->csize)
        {
            //Get the next cluster.
            temp = get_fat(fs->curr_clust);
            //Check the EOC mark.
            if (is_eoc(temp) == 0)
            {
                if ((temp <= 1) || (temp >= fs->max_clust))
                    goto fr_abort;
                fs->curr_clust = temp;
                //Get the first sector of the cluster.
                temp = clust2sect(temp);
                if (temp == 0)
                    goto fr_abort;
                fs->dsect = temp;
                fs->csect = 0;
            }
        }
    }

    return FR_OK;

fr_abort:
    fs->flag = 0;
    return FR_DISK_ERR;
}

#endif //FATFS_USE_READ


//******************************************************************************
//* Write data to the opened file.                                             *
//******************************************************************************
#if FATFS_USE_WRITE

unsigned char pf_write(
    const void     *buff,   //Pointer to the data to be written.
    unsigned short btw,     //Number of bytes to write.
    unsigned short *bw      //Pointer to number of bytes written.
)
{
    unsigned short bcnt, bofs, scnt, sofs;
    unsigned long  temp;
    unsigned char  *wrbuff = (unsigned char *)buff;
    PFATFS         fs = FatFs;

    //Clear the number of bytes written.
    *bw = 0;
    
    //Check if the volume is mounted.
    if (fs == 0)
        return FR_NOT_ENABLED;
    //Check if the file is opened.
    if ((fs->flag & FA_OPENED) == 0)
        return FR_NOT_OPENED;

    //Calculate the remaining bytes.
    temp = fs->fsize - fs->fptr;
    if ((unsigned long)btw > temp)
        btw = (unsigned short)temp;

    //Read the data.
    while (btw > 0)
    {
        //Check whether the file pointer is at sector boundary.
        bofs = (unsigned short)fs->fptr % 512;
        if ((bofs > 0) || (btw < 512))
        {
            //Calculate the number of bytes to write.
            bcnt = 512 - bofs;
            if (btw < bcnt)
                bcnt = btw;
            //Increment the sector offset when the sector boundary is reached.
            scnt = (bofs + bcnt) / 512;
            //Read the sector to be overwritten.
            if (DiskReadPartialSector(fs->dsect, 0, 0, 512) != DISK_OK)
                goto fw_abort;
            //Write the data to the sector buffer.
            if (MemCardWritePartialSector(0, wrbuff, bofs, bcnt) != DISK_OK)
                goto fw_abort;
            //Finalize the write operation.
            if (MemCardWritePartialSector(fs->dsect, 0, 0, 512) != DISK_OK)
                goto fw_abort;
        }
        else
        {
            //Calculate the number of sectors to write.
            sofs = ((unsigned short)fs->fptr / 512) & (unsigned short)(fs->csize - 1);
            scnt = (unsigned short)fs->csize - sofs;
            sofs = btw / 512;
            if (scnt > 127)
                scnt = 127;
            if (scnt > sofs)
                scnt = sofs;
            //Number of bytes to write.
            bcnt = scnt * 512;
            //Write whole sectors.
            if (MemCardWriteMultipleSectors(fs->dsect, wrbuff, (unsigned char)scnt) != DISK_OK)
                goto fw_abort;
        }

        //Update the pointers.
        fs->csect += scnt;
        fs->dsect += scnt;
        fs->fptr += bcnt;
        wrbuff += bcnt;
        btw -= bcnt;
        *bw += bcnt;

        //Check whether the cluster boundary has been reached.
        if (fs->csect >= fs->csize)
        {
            //Get the next cluster.
            temp = get_fat(fs->curr_clust);
            //Check the EOC mark.
            if (is_eoc(temp) == 0)
            {
                if ((temp <= 1) || (temp >= fs->max_clust))
                    goto fw_abort;
                fs->curr_clust = temp;
                //Get the first sector of the cluster.
                temp = clust2sect(temp);
                if (temp == 0)
                    goto fw_abort;
                fs->dsect = temp;
                fs->csect = 0;
            }
        }
    }

    return FR_OK;

fw_abort:
    fs->flag = 0;
    return FR_DISK_ERR;
}

#endif //FATFS_USE_WRITE


//******************************************************************************
//* Moves the file read/write pointer to the specified location.               *
//******************************************************************************
#if FATFS_USE_LSEEK

unsigned char pf_lseek(unsigned long ofs)
{
    unsigned long ifptr, bcs, clst, new_clst, sect;
    unsigned char eoc = 0;
    PFATFS        fs = FatFs;

    //Check if the volume is mounted.
    if (fs == 0)
        return FR_NOT_ENABLED;
    //Check if the file is opened.
    if ((fs->flag & FA_OPENED) == 0)
        return FR_NOT_OPENED;

    //Clip the offset if it is greater than the file size.
    if (ofs > fs->fsize)
        ofs = fs->fsize;

    //Reset the file pointer.
    ifptr = fs->fptr;
    fs->fptr = 0;

    if (ofs > 0) 
    {
        //Cluster size in bytes.
        bcs = (unsigned long)fs->csize * 512;
        //Current cluster offset multiplied with the number of bytes per cluster.
        if ((ifptr > 0) && (ofs == fs->fsize))
            ifptr--;
        clst = ifptr & ~(bcs - 1);
        //New cluster offset multiplied with the number of bytes per cluster.
        new_clst = ofs & ~(bcs - 1);

        //Determine the start cluster.
        if (new_clst >= clst)
        {
            //When seek forward, start from the current cluster.
            fs->fptr = clst;
            ofs -= clst;
            clst = fs->curr_clust;
        }
        else 
        {
            //When seek backward, start from the first cluster.
            clst = fs->org_clust;			
            fs->curr_clust = clst;
        }

        //Cluster following loop.
        while (ofs >= bcs) 
        {
            //Get the next cluster.
            clst = get_fat(clst);
            eoc = is_eoc(clst);
            if (eoc == 0)
            {
                if ((clst <= 1) || (clst >= fs->max_clust))
                    goto lseek_abort;
                fs->curr_clust = clst;
            }
            //Update the file pointer and the offset.
            fs->fptr += bcs;
            ofs -= bcs;
        }

        //Increment the file pointer with the byte offset.
        fs->fptr += ofs;
    }
    else
    {
        //Set the first cluster.
        clst = fs->org_clust;			
        fs->curr_clust = clst;
    }

    //Sector offset in the cluster.
    fs->csect = (unsigned char)((unsigned short)ofs / 512);
    if (eoc == 0)
    {
        //Get the start sector of the current cluster.
        sect = clust2sect(clst);
        if (sect == 0)
            goto lseek_abort;
        //Current data sector.
        fs->dsect = sect + fs->csect;
    }

    return FR_OK;

lseek_abort:
    fs->flag = 0;
    return FR_DISK_ERR;
}

#endif //FATFS_USE_LSEEK


//******************************************************************************
//* Creates a directory object.                                                *
//******************************************************************************
#if FATFS_USE_DIR

unsigned char pf_opendir(PDIR dj, const char *path)
{
    unsigned char res;
    unsigned char buff[12];
    FAT_DIR_ENTRY dir;
    PFATFS        fs = FatFs;

    //Check if the volume is mounted.
    if (fs == 0)
    {
        res = FR_NOT_ENABLED;
    }
    else 
    {
        fs->buf = (unsigned char *)&dir;
        dj->fn = buff;

        //Follow the path to the directory.
        res = follow_path(dj, path);
        if (res == FR_OK)
        {
            //Check if it is the root directory.
            if (dir.DIR_Name[0] != 0) 
            {
                //It is not the root directory. Check if it is a directory.
                if ((dir.DIR_Attr & AM_DIR) != 0)
                {
                    //The entry is a directory.
                    dj->sclust = ((unsigned long)LD_WORD(&dir.DIR_FstClusHI) << 16) | LD_WORD(&dir.DIR_FstClusLO);
                }
                else 
                {
                    //The entry is not a directory.
                    res = FR_NO_PATH;
                }
            }
            
            if (res == FR_OK)
                res = dir_rewind(dj);
        }
        
        if (res == FR_NO_FILE)
            res = FR_NO_PATH;
    }

    return res;
}


//******************************************************************************
//* Reads the next directory entry.                                            *
//******************************************************************************
unsigned char pf_readdir(PDIR dj, FILINFO *fno)
{
    unsigned char res;
    unsigned char buff[12];
    FAT_DIR_ENTRY dir;
    PFATFS        fs = FatFs;

    //Check if the volume is mounted.
    if (fs == 0)
    {
        res = FR_NOT_ENABLED;
    }
    else 
    {
        fs->buf = (unsigned char *)&dir;
        dj->fn = buff;

        if (fno == 0)
        {
            //Set the directory pointer to the root directory.
            res = dir_rewind(dj);
        }
        else 
        {
            //Read the next directory entry.
            res = dir_read(dj);
            if (res == FR_NO_FILE)
            {
                dj->sect = 0;
                res = FR_OK;
            }
            if (res == FR_OK)
            {
                //A valid entry has been found. Get the object information.
                get_fileinfo(dj, fno);
                //Set the directory pointer to the next entry.
                res = dir_next(dj);
                if (res == FR_NO_FILE)
                {
                    dj->sect = 0;
                    res = FR_OK;
                }
            }
        }
    }

    return res;
}

#endif //FATFS_USE_DIR
