//******************************************************************************
//* Driver for the EADOGS102 LCD module.                                       *
//* Written by   : Tamas Raikovich                                             *
//* Version      : 1.0                                                         *
//* Last modified: 01/10/2011                                                  *
//******************************************************************************
#include <xparameters.h>
#include "lcd.h"
#include "..\spi\spi.h"
#include "image_data.h"


//******************************************************************************
//* Constant definitions.                                                      *
//******************************************************************************
#define LCD_CLK_FREQ_HZ         15000000
#define LCD_DATA_MASK           0x00ff
#define LCD_DATA_CMD            (1 << 8)


//******************************************************************************
//* Write a data byte to the LCD SRAM.                                         *
//******************************************************************************
void LCDWriteData(unsigned char data)
{
    SPIReadWriteByte((unsigned long)data | LCD_DATA_CMD);
}


//******************************************************************************
//* Send a command to the LCD.                                                 *
//******************************************************************************
void LCDSendCmd(unsigned char cmd)
{
    SPIReadWriteByte((unsigned long)cmd);
}


//******************************************************************************
//* Initializes the LCD.                                                       *
//******************************************************************************
void LCDInitialize(unsigned long topView)
{
    unsigned long i, j;
    unsigned char segDir = (topView) ? 0xa0 : 0xa1;
    unsigned char comDir = (topView) ? 0xc8 : 0xc0;

    SPISelect(SPI_DEVICE_LCD, LCD_CLK_FREQ_HZ);

    LCDSendCmd(0x40);       //Set scroll line to 0.
    LCDSendCmd(segDir);     //Set SEG direction.
    LCDSendCmd(comDir);     //Set COM direction.
    LCDSendCmd(0xa4);       //Set all pixels on.
    LCDSendCmd(0xa6);       //Set display inverse off.
    LCDSendCmd(0xa2);       //Set LCD bias to 1/9 (duty cycle: 1/65).
    LCDSendCmd(0x2f);       //Enable booster, regulator and follower.
    LCDSendCmd(0x27);       //Set the contrast.
    LCDSendCmd(0x81);
    LCDSendCmd(0x08);
    LCDSendCmd(0xfa);       //Set temperature compensation to -0.11%/C
    LCDSendCmd(0x90);

    //Clear the SRAM.
    for (i = 0; i < 9; i++)
    {
        //Set the page and column address.
        LCDSendCmd(0xb0 | (i & 0x0f));
        LCDSendCmd(0x00);
        LCDSendCmd(0x10);

        //Fill the current page with zeroes.
        for (j = 0; j < 132; j++)
            LCDWriteData(0x00);
    }

    LCDSendCmd(0xaf);       //Set display enable.

    SPIDeselect();
}


//******************************************************************************
//* Sets the page and the column address.                                      *
//******************************************************************************
void LCDSetAddress(unsigned char page, unsigned char column)
{
    SPISelect(SPI_DEVICE_LCD, LCD_CLK_FREQ_HZ);

    LCDSendCmd(0xb0 | (page & 0x0f));               //Set the page address.
    LCDSendCmd(0x00 | (column & 0x0f));             //Set the column address.
    LCDSendCmd(0x10 | ((column >> 4) & 0x0f));

    SPIDeselect();
}


//******************************************************************************
//* Sets the LCD contrast.                                                     *
//******************************************************************************
void LCDSetContrast(unsigned char contrast)
{
    SPISelect(SPI_DEVICE_LCD, LCD_CLK_FREQ_HZ);

    LCDSendCmd(0x81);                               //Set the LCD contrast
    LCDSendCmd(0x00 | (contrast & 0x3f));

    SPIDeselect();
}


//******************************************************************************
//* Draws an image to the LCD display.                                         *
//******************************************************************************
void LCDDrawTestImage()
{
    unsigned long page;
    unsigned long column;

    //Initialize the LCD module.
    LCDInitialize(1);

    //Draw something on the LCD.
    for (page = 0; page < 8; page++)
    {
        //Set the page and column address.
        LCDSetAddress(page, 30);

	//Set LCD chip select low.
        SPISelect(SPI_DEVICE_LCD, LCD_CLK_FREQ_HZ);

        for (column = 0; column < 102; column++)
        {
            LCDWriteData(image_data[102*page+column]);
	}

        //Set LCD chip select high.
        SPIDeselect();
    }
}

